"""
 Copyright 2023 [Anonymized]
 
 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

      https://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 """

from typing import Any, Dict, List, Optional

import tensorflow as tf
from tensorflow import Tensor
from tensorflow.keras import layers
from tensorflow_similarity.models import SimilarityModel
from keras_nlp.layers import TransformerEncoder


# from tensorflow_retvec import RetVecBinarizer
from .layers import pooling

from .outputs import build_similarity_output

from retvec.tf.models.layers import get_pooling_layer, dense_block
from retvec.tf.layers import RETVecBinarizer, RETVecIntegerizer, RETVecIntToBinary
from retvec.tf.models.positional_embeddings import (
    ScaledSinusoidalPositionalEmbedding,
    PositionalEmbedding,
)
from retvec.tf.models.gau import GAU

from .t5 import T5Block
from .layers import ConvNextBlock


def build_retsim_model_from_config(config: Dict) -> tf.keras.Model:
    m = config["model"]
    return build_retsim_model(
        max_len=m["max_len"],
        char_encoding_size=m["char_encoding_size"],
        char_encoding_type=m["char_encoding_type"],
        model_type=m["model_type"],
        encoder_hidden_dim=m["encoder_hidden_dim"],
        encoder_abs_position_encoding_type=m["encoder_abs_position_encoding_type"],
        encoder_blocks=m["encoder_blocks"],
        encoder_shared_dim=m["encoder_shared_dim"],
        encoder_expansion_factor=m["encoder_expansion_factor"],
        encoder_activation=m["encoder_activation"],
        encoder_attention_activation=m["encoder_attention_activation"],
        encoder_norm_type=m["encoder_norm_type"],
        encoder_position_encoding_type=m["encoder_position_encoding_type"],
        encoder_dropout=m["encoder_dropout"],
        encoder_attention_dropout=m["encoder_attention_dropout"],
        encoder_spatial_dropout=m["encoder_spatial_dropout"],
        encoder_epsilon=m["encoder_epsilon"],
        encoder_num_heads=m.get("encoder_num_heads", 4),
        pooling_dropout=m["pooling_dropout"],
        pooling_type=m["pooling_type"],
        similarity_dim=m["similarity_dim"],
        similarity_dense_layers=m["similarity_dense_layers"],
        similarity_dense_contraction_rate=m["similarity_dense_contraction_rate"],
    )


def build_retsim_model(
    max_len: int = 512,
    char_encoding_size: int = 24,
    char_encoding_type: str = "UTF-8",
    model_type: str = "gau",  # GAU, T5, BERT, MLP, CNN supported
    encoder_hidden_dim: int = 128,
    encoder_abs_position_encoding_type: str = "scaled_sin",
    encoder_blocks: int = 2,
    encoder_shared_dim: int = 128,
    encoder_expansion_factor: int = 1,
    encoder_activation: str = "swish",
    encoder_attention_activation: str = "sqrrelu",
    encoder_norm_type: str = "scaled",
    encoder_position_encoding_type: str = "rope",
    encoder_dropout: float = 0.0,
    encoder_attention_dropout: float = 0.05,
    encoder_spatial_dropout: float = 0.0,
    encoder_epsilon: float = 1e-6,
    encoder_num_heads: int = 4,  # t5 and bert only
    pooling_dropout: float = 0.0,
    pooling_type: str = "GEM",
    similarity_dim: int = 128,
    similarity_dense_layers: int = 1,
    similarity_dense_contraction_rate: int = 1,
) -> tf.keras.Model:
    inputs = layers.Input(shape=(1,), name="token", dtype=tf.string)

    # character embedding
    # encoder = RETVecBinarizer(
    #     max_len,
    #     encoding_size=char_encoding_size,
    #     encoding_type=char_encoding_type,
    #     name="binarizer",
    # )(inputs)
    encoder = RETVecIntegerizer(max_len, encoding_type=char_encoding_type, name="integerizer")(inputs)
    encoder = RETVecIntToBinary(
        sequence_length=1,
        word_length=max_len,
        encoding_size=char_encoding_size,
        name="binarizer",
    )(encoder)

    if encoder_abs_position_encoding_type == "scaled_sin":
        encoder = ScaledSinusoidalPositionalEmbedding(hidden_size=char_encoding_size)(
            encoder
        )

    elif encoder_abs_position_encoding_type == "abs":
        encoder = PositionalEmbedding()(encoder)

    encoder = layers.Dense(encoder_hidden_dim, name="encoder_start")(encoder)

    # check if we need batch norm and activation here? or maybe it should be a layer norm...
    # encoder = layers.BatchNormalization()(encoder)
    encoder = layers.Activation(encoder_activation)(encoder)

    for _ in range(encoder_blocks):
        if model_type == "gau":
            encoder = GAU(
                dim=encoder_hidden_dim,
                max_len=max_len,
                shared_dim=encoder_shared_dim,
                expansion_factor=encoder_expansion_factor,
                activation=encoder_activation,
                attention_activation=encoder_attention_activation,
                position_encoding_type=encoder_position_encoding_type,
                dropout_rate=encoder_dropout,
                norm_type=encoder_norm_type,
                attention_dropout_rate=encoder_attention_dropout,
                spatial_dropout_rate=encoder_spatial_dropout,
                epsilon=encoder_epsilon,
            )(encoder)

        elif model_type == "t5":
            encoder_num_heads = 4
            hidden_dim = encoder_hidden_dim * encoder_expansion_factor
            head_dim = hidden_dim // encoder_num_heads
            encoder = T5Block(
                dim=encoder_hidden_dim,
                hidden_dim=hidden_dim,
                num_heads=encoder_num_heads,
                head_dim=head_dim,
                activation=encoder_activation,
                dropout_rate=encoder_dropout,
                use_gated_ffn=True,
                norm_type="t5",
            )(encoder)

        elif model_type == "bert":
            encoder = TransformerEncoder(
                intermediate_dim=encoder_hidden_dim,
                num_heads=4,
                dropout=encoder_dropout,
                activation=encoder_activation,
                layer_norm_epsilon=encoder_epsilon,
            )(encoder)

        elif model_type == "cnn":
            encoder = ConvNextBlock(
                kernel_size=7,
                depth=2,
                filters=encoder_hidden_dim,
                hidden_dim=encoder_hidden_dim * encoder_expansion_factor,
                dropout_rate=encoder_dropout,
                activation="gelu"
            )(encoder)

        elif model_type == "dense":
            encoder = dense_block(encoder,
                                  units=encoder_hidden_dim,
                                  activation="gelu",
                                  norm_type=encoder_norm_type)

    encoder = get_pooling_layer(encoder, pooling_type=pooling_type)

    encoder = layers.Dropout(pooling_dropout)(encoder)

    outputs = build_similarity_output(
        encoder=encoder,
        activation=encoder_activation,
        similarity_dim=similarity_dim,
        dense_layers=similarity_dense_layers,
        dense_contraction_rate=similarity_dense_contraction_rate,
    )

    return SimilarityModel(inputs, outputs)
