import os
from madacbench.sigmoid.global_state_env import SigmoidMultiActMultiValActionState
import argparse
import torch

from tianshou.data import Collector, VectorReplayBuffer
from tianshou.env import DummyVectorEnv
from tianshou.policy import DQNPolicy
from tianshou.trainer import offpolicy_trainer
from tianshou.utils import TensorboardLogger
from tianshou.utils.net.common import Net
from torch.utils.tensorboard import SummaryWriter

def get_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--key', type=str, default='Sigmoid')
    parser.add_argument('--seed', type=int, default=2023)
    parser.add_argument('--n_steps', type=int, default=10)
    parser.add_argument('--n_agents', type=int, default=2)
    parser.add_argument('--n_actions', type=int, default=10)
    args = parser.parse_known_args()[0]
    return args


class DQNargs:
    train_num = 1
    test_num = 1
    hidden_sizes = [128, 128, 128]
    device = 'cuda' if torch.cuda.is_available() else 'cpu'
    lr = 3e-4
    gamma = 0.99
    n_step = 3
    target_update_freq = 160
    buffer_size = 50000
    eps_train = 0.1
    epoch = 1
    step_per_epoch = 1050000
    step_per_collect = 10
    update_per_step = 0.05
    batch_size = 32
    episode_per_test = 100


if __name__ == "__main__":
    args = get_args()
    args_test = get_args()
    dqn_args = DQNargs()
    env = SigmoidMultiActMultiValActionState(**vars(args))
    train_envs = DummyVectorEnv(
        [lambda: SigmoidMultiActMultiValActionState(**vars(args)) for _ in range(dqn_args.train_num)])
    args_test.key = "Sigmoid"
    test_envs = DummyVectorEnv(
        [lambda: SigmoidMultiActMultiValActionState(**vars(args_test)) for _ in range(dqn_args.test_num)])
    seed_list_train = [args.seed for _ in range(dqn_args.train_num)]
    seed_list_test = [args.seed for _ in range(dqn_args.test_num)]
    state_shape = env.observation_space.shape or env.observation_space.n
    # action_shape = env.action_space.shape or env.action_space.n
    action_shape = env.n_actions ** env.n_agents
    net = Net(state_shape=state_shape, action_shape=action_shape, hidden_sizes=dqn_args.hidden_sizes,
              device=dqn_args.device).to(dqn_args.device)
    optim = torch.optim.Adam(net.parameters(), lr=dqn_args.lr)
    policy = DQNPolicy(net, optim, discount_factor=dqn_args.gamma,
                       estimation_step=dqn_args.n_step, target_update_freq=dqn_args.target_update_freq)
    buf = VectorReplayBuffer(dqn_args.buffer_size, buffer_num=len(train_envs))
    train_collector = Collector(
        policy, train_envs, buf, exploration_noise=True)
    test_collector = Collector(policy, test_envs, exploration_noise=False)
    log_path = os.path.join('./results/dqn', args.key)
    writer = SummaryWriter(log_path)
    logger = TensorboardLogger(writer)

    def save_best_fn(policy):
        torch.save(policy.state_dict(), os.path.join(
            log_path, args.key + 'policy.pth'))

    def train_fn(epoch, env_step):
        # eps annnealing, just a demo
        if env_step <= 100000:
            policy.set_eps(dqn_args.eps_train)
        elif env_step <= 500000:
            eps = dqn_args.eps_train - (env_step - 100000) / \
                400000 * (0.9 * dqn_args.eps_train)
            policy.set_eps(eps)
        else:
            policy.set_eps(0.1 * dqn_args.eps_train)


    def save_checkpoint_fn(epoch, env_step, gradient_step):
        for w in train_envs.workers:
            w.env.save_replay('dqn')

    result = offpolicy_trainer(
        policy,
        train_collector,
        test_collector,
        max_epoch=dqn_args.epoch,
        step_per_epoch=dqn_args.step_per_epoch,
        step_per_collect=dqn_args.step_per_collect,
        update_per_step=dqn_args.update_per_step,
        episode_per_test=dqn_args.episode_per_test,
        batch_size=dqn_args.batch_size,
        train_fn=train_fn,
        save_fn=save_best_fn,
        save_checkpoint_fn=save_checkpoint_fn,
        logger=logger
    )
    print(f'Finished training! Use {result["duration"]}')
