import functools

import jax
from jax import nn as jnn
from jax import numpy as jnp
from jax import random as jrandom

from neural_networks_chomsky_hierarchy.tasks import task

class Last(task.GeneralizationTask):
  """Ends with DCB

  Examples:
    A -> class 0
    BCD, ABCD -> class 1

  Note the sampling is jittable so this task is fast.
  """

  def __init__(self, *args, **kwargs):
    """Initializes the remember_string task.

    Args:
      vocab_size: The size of the alphabet.
      *args: Args for the base task class.
      **kwargs: Kwargs for the base task class.
    """
    super().__init__(*args, **kwargs)

  @functools.partial(jax.jit, static_argnums=(0, 2, 3))
  def sample_batch(self, rng: jnp.ndarray, batch_size: int,
                   length: int) -> task.Batch:
    """Returns a batch of strings and the expected class."""
    reserve = batch_size // 2
    strings = jrandom.randint(
        rng,
        shape=(batch_size, length),
        minval=0,
        maxval=2,
    )
    strings = strings.at[:reserve, -1].set(1)
    strings = strings.at[reserve:, -1].set(0)
    one_hot_strings = jnn.one_hot(strings, num_classes=2)

    labels = strings[:,-1] == 1
    labels = labels.astype(jnp.float32)
    labels = jnn.one_hot(labels, num_classes=2)
    return {
        'input': one_hot_strings,
        'output': labels,
    }

  @property
  def input_size(self) -> int:
    """Returns the input size for the models."""
    return self._vocab_size

  @property
  def output_size(self) -> int:
    """Returns the output size for the models."""
    return 2