import functools

import jax
from jax import nn as jnn
from jax import numpy as jnp
from jax import random as jrandom

from neural_networks_chomsky_hierarchy.tasks import task


class Even(task.GeneralizationTask):
  """(AB)*

  Examples:
    A, ABA -> class 0
    AB, ABAB -> class 1

  Note the sampling is jittable so this task is fast.
  """

  @functools.partial(jax.jit, static_argnums=(0, 2, 3))
  def sample_batch(self, rng: jnp.ndarray, batch_size: int,
                   length: int) -> task.Batch:
    """Returns a batch of strings and the expected class."""

    strings = jnp.zeros((batch_size, length))
    if length % 2 == 1:
        strings = strings.at[:batch_size // 2, length // 2 + 1:].set(1)
        strings = strings.at[batch_size // 2:, length // 2:].set(1)
    else:
        strings = strings.at[:,length // 2:].set(1)
    strings = jrandom.permutation(rng, strings, axis=1, independent=True)
    one_hot_strings = jnn.one_hot(strings, num_classes=2)

    labels = (strings == 0).sum(axis=-1) == (strings == 1).sum(axis=-1)
    labels = labels.astype(jnp.float32)
    labels = jnn.one_hot(labels, num_classes=2)
    return {
        'input': one_hot_strings,
        'output': labels,
    }

  @property
  def input_size(self) -> int:
    """Returns the input size for the models."""
    return 2

  @property
  def output_size(self) -> int:
    """Returns the output size for the models."""
    return 2