import functools

import jax
from jax import nn as jnn
from jax import numpy as jnp
from jax import random as jrandom

from neural_networks_chomsky_hierarchy.tasks import task


class Star(task.GeneralizationTask):
  """(AA)*

  Examples:
    A, BB -> class 0
    AA -> class 1

  Note the sampling is jittable so this task is fast.
  """

  @functools.partial(jax.jit, static_argnums=(0, 2, 3))
  def sample_batch(self, rng: jnp.ndarray, batch_size: int,
                   length: int) -> task.Batch:
    """Returns a batch of strings and the expected class."""
    reserve = batch_size // 2
    strings = jrandom.randint(
        rng,
        shape=(batch_size - reserve, length),
        minval=0,
        maxval=2,
    )
    if reserve > 0:
      strings = jnp.concatenate((jnp.zeros((reserve, length)), strings), axis=0)
    one_hot_strings = jnn.one_hot(strings, num_classes=2)
    if length % 2 == 0:
      labels = jnp.all(strings == 0, axis=1).astype(jnp.float32)
    else:
      labels = jnp.zeros((batch_size))
    labels = jnn.one_hot(labels, num_classes=2)
    return {
        'input': one_hot_strings,
        'output': labels,
    }

  @property
  def input_size(self) -> int:
    """Returns the input size for the models."""
    return 2

  @property
  def output_size(self) -> int:
    """Returns the output size for the models."""
    return 2
