import argparse
import json
from datetime import datetime
import os
import pathlib
from tqdm import tqdm

import torch
import torch.nn.functional as F
import numpy as np
import pandas as pd
from scipy.stats import spearmanr

from sequence_models.collaters import StructureCollater, SimpleCollater
from sequence_models.gnn import BidirectionalStruct2SeqDecoder
from sequence_models.constants import PROTEIN_ALPHABET, PAD, MASK
from sequence_models.convolutional import ByteNetLM
from sequence_models.utils import parse_fasta, Tokenizer
from sequence_models.pdb_utils import parse_PDB, process_coords


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('config_fpath', type=str, help='file path to config json for model')
    parser.add_argument('--map_dir', type=str, default=os.getenv('PT_MAP_OUTPUT_DIR', '/tmp') + '/')
    parser.add_argument('--out_fpath', type=str, required=False, default=os.getenv('PT_OUTPUT_DIR', '/tmp') + '/')
    parser.add_argument('--gpu', '-g', type=int, default=0)
    parser.add_argument('--cnn', action='store_true')
    parser.add_argument('--no_gnn', action='store_true')
    parser.add_argument('--af2', action='store_true')
    parser.add_argument('--use_wt', action='store_true')
    parser.add_argument('--cnn', action='store_true')



    args = parser.parse_args()
    args.world_size = 1
    train(0, args)


def train(gpu, args):
    _ = torch.manual_seed(32)
    torch.cuda.set_device(gpu + args.gpu)
    device = torch.device('cuda:' + str(gpu + args.gpu))

    ## Grab metadata
    try:
        data_fpath = os.getenv('PT_DATA_DIR') + '/'
    except:
        home = str(pathlib.Path.home())
        data_fpath = home + '/data/'
    af2_fpath = data_fpath + 'af2/'
    pdb_fpath = data_fpath + 'pdb/'
    data_fpath += 'gb1/'

    # grab model hyperparameters
    with open(args.config_fpath, 'r') as f:
        config = json.load(f)
    n_tokens = len(PROTEIN_ALPHABET)
    d_model = config['d_model']
    node_features = 10
    edge_features = 11
    dropout = config['dropout']
    use_mpnn = config['use_mpnn']
    n_layers = config['n_layers']
    n_connections = config['n_connections']

    # Prep results
    out_fname = 'gb1.csv'
    if out_fname in os.listdir(args.out_fpath):
        record = pd.read_csv(args.out_fpath + out_fname)
    else:
        record = pd.read_csv(args.map_dir + 'metrics.csv', header=None)
        if args.no_gnn:
            record.columns = ['loss', 'accuracy', 'tokens', 'step']
            temp = pd.DataFrame(np.array([[np.nan, np.nan, 0, 0]]), columns=record.columns)
        else:
            record.columns = ['step', 'loss']
            record['epoch'] = record.index
            record['epoch'] += 1
            temp = pd.DataFrame(np.array([[0, np.nan, 0]]), columns=record.columns)
        # min_idx = record.sort_values('loss').index
        # record = record.loc[:min_idx[0]]
        # first_loss = record.iloc[0, 1]
        # last_loss = record.iloc[-1, 1]
        # bins = np.linspace(first_loss, last_loss, 10)[1:-1]
        # j = 0
        # idx = []
        # for i, row in record.iterrows():
        #     if row['loss'] < bins[j]:
        #         j += 1
        #         idx.append(i)
        #         if j == len(bins):
        #             break
        # record = pd.concat([record.iloc[-1:], temp, record.iloc[[0] + idx]], ignore_index=True)
        record = pd.concat([record.iloc[np.arange(len(record) - 1, -1, -1)], temp], ignore_index=True)
        record = record.astype({'step': int})
    # Get the wt sequence, locations, and mutations
    pdb = '2gi9'
    locations = [38, 39, 40, 53]
    wt_code = list('VDGV')
    df = pd.read_csv(data_fpath + 'four_mutations_full_data.csv').iloc[1:]
    df = df.reset_index()
    tgt = df['Fitness'].values
    n_mutations = np.array([sum([s1 != s2 for s1, s2 in zip(row['Variants'], wt_code)]) for i, row in df.iterrows()])
    one_idx = (n_mutations == 1)
    two_idx = (n_mutations == 2)
    three_idx = (n_mutations == 3)
    four_idx = (n_mutations == 4)


    # preprocess the structure just once
    collater = StructureCollater(SimpleCollater(PROTEIN_ALPHABET), n_connections=n_connections)
    tkn = Tokenizer(alphabet=PROTEIN_ALPHABET)
    if args.af2:
        coords, wt, _ = parse_PDB(af2_fpath + 'gb1_a60fb_unrelaxed_rank_1_model_5.pdb')
    else:
        coords, wt, _ = parse_PDB(pdb_fpath + pdb + '.pdb')
    tokenized = torch.tensor(tkn.tokenize(wt))
    coords = {
        'N': coords[:, 0],
        'CA': coords[:, 1],
        'C': coords[:, 2]
    }
    dist, omega, theta, phi = process_coords(coords)
    batch = [[wt, torch.tensor(dist, dtype=torch.float),
              torch.tensor(omega, dtype=torch.float),
              torch.tensor(theta, dtype=torch.float), torch.tensor(phi, dtype=torch.float)]]
    _, nodes, edges, connections, edge_mask = collater(batch)
    nodes = nodes.to(device)
    edges = edges.to(device)
    connections = connections.to(device)
    edge_mask = edge_mask.to(device)

    for idx, row in record.iterrows():
        step = str(int(row['step']))
        wd = args.map_dir + 'checkpoint' + step + '.tar'
        if args.cnn:
            d_cnn = 1280
            n_cnn_layers = 56
            kernel_size = 5
            r = 128
            slim = False
            d_embed = 8
            activation = 'gelu'
            causal = False
            pad_idx = PROTEIN_ALPHABET.index(PAD)
            cnn = ByteNetLM(n_tokens, d_embed, d_cnn, n_cnn_layers, kernel_size, r, final_ln=True,
                                slim=slim, activation=activation, causal=causal, padding_idx=pad_idx).to(device)

        if not args.no_gnn:
            gnn = BidirectionalStruct2SeqDecoder(n_tokens, node_features, edge_features,
                                               d_model, num_decoder_layers=n_layers,
                                               dropout=dropout, use_mpnn=use_mpnn,
                                                 pe=False, one_hot_src=(not args.cnn)).to(device)
        if int(row['step'] != 0):
            sd = torch.load(wd, map_location=device)
            if args.cnn and args.no_gnn:
                sd = sd['model_state_dict']
                sd = {k.split('module.')[1]: v for k, v in sd.items()}
                cnn.load_state_dict(sd)
                cnn = cnn.to(device).eval()
            elif args.cnn:
                gnn_sd = sd['decoder_state_dict']
                gnn.load_state_dict(gnn_sd)
                gnn = gnn.to(device).eval()
                cnn_sd = sd['encoder_state_dict']
                cnn_sd = {k.split('module.')[1]: v for k, v in cnn_sd.items()}
                cnn.load_state_dict(cnn_sd)
                cnn = cnn.to(device).eval()
            else:
                sd = sd['model_state_dict']
                sd = {k.split('module.')[1]: v for k, v in sd.items()}
                gnn.load_state_dict(sd)
                gnn = gnn.to(device).eval()

        pseudo_likelihood = {}
        pre = np.zeros(len(df))
        for i, row in tqdm(df.iterrows(), total=len(df)):
            muts = row['Variants']
            muts = [(pos, m) for m, n, pos in zip(muts, wt_code, locations) if m != n]
            aas = [PROTEIN_ALPHABET.index(m[1]) for m in muts]
            pos = [m[0] for m in muts]
            pos = tuple(pos)
            seq = ''
            start = 0
            for p in pos:
                seq += wt[start:p]
                start = p + 1
                seq += MASK
            seq += wt[start:]
            src = torch.tensor(tkn.tokenize(seq)).unsqueeze(0).to(device)
            if args.use_wt:
                if pos not in pseudo_likelihood:
                    with torch.no_grad():
                        if args.cnn:
                            src = cnn(src)
                            if args.no_gnn:
                                output = F.log_softmax(src, dim=-1)
                        if not args.no_gnn:
                            output = F.log_softmax(gnn(nodes, edges, connections, src, edge_mask), dim=-1)
                        pl = output[0, pos].cpu()
                        wt_pl = output[torch.zeros(len(pos)).long(), pos, tokenized[list(pos)]].cpu()
                        pseudo_likelihood[pos] = (pl - wt_pl.unsqueeze(-1)).numpy()
                pre[i] = pseudo_likelihood[pos][np.arange(len(pos)), np.array(aas)].sum()
            else:
                with torch.no_grad():
                    if args.cnn:
                        src = cnn(src)
                        if args.no_gnn:
                            output = F.log_softmax(src, dim=-1)
                    if not args.no_gnn:
                        output = F.log_softmax(gnn(nodes, edges, connections, src, edge_mask), dim=-1) # 1, ell, t
                    # Slice out correct log probs
                    src = torch.tensor(tkn.tokenize(seq)).unsqueeze(0).to(device)
                    _, ell = src.shape
                    output = output[0, np.arange(ell), src[0]]
                    # Add them up
                    pre[i] = output.mean().cpu().item()

        rho = spearmanr(pre, tgt)
        record.loc[idx, 'gb1_rho'] = rho.correlation
        rho = spearmanr(pre[one_idx], tgt[one_idx])
        record.loc[idx, 'gb1_1_rho'] = rho.correlation
        rho = spearmanr(pre[two_idx], tgt[two_idx])
        record.loc[idx, 'gb1_2_rho'] = rho.correlation
        rho = spearmanr(pre[three_idx], tgt[three_idx])
        record.loc[idx, 'gb1_3_rho'] = rho.correlation
        rho = spearmanr(pre[four_idx], tgt[four_idx])
        record.loc[idx, 'gb1_4_rho'] = rho.correlation
        print(record)
        record.to_csv(args.out_fpath + out_fname, index=False)
        # Save predictions
        predictions = pd.DataFrame()
        predictions['variant'] = df['Variants']
        predictions['n_mutations'] = n_mutations
        predictions['tgt'] = tgt
        predictions['pre'] = pre
        predictions.to_csv(args.out_fpath + 'predictions%s.csv' %step, index=False)

if __name__ == '__main__':
    main()