import argparse
import json
from datetime import datetime
import os
import pathlib
from tqdm import tqdm

import torch
import torch.nn.functional as F
import numpy as np
import pandas as pd
from scipy.stats import spearmanr, pearsonr

from sequence_models.collaters import StructureCollater, SimpleCollater
from sequence_models.gnn import BidirectionalStruct2SeqDecoder
from sequence_models.constants import PROTEIN_ALPHABET, PAD, MASK
from sequence_models.convolutional import ByteNetLM
from sequence_models.utils import parse_fasta, Tokenizer
from sequence_models.pdb_utils import parse_PDB, process_coords


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('config_fpath', type=str, help='file path to config json for model')
    parser.add_argument('--map_dir', type=str, default=os.getenv('PT_MAP_OUTPUT_DIR', '/tmp') + '/')
    parser.add_argument('--out_fpath', type=str, required=False, default=os.getenv('PT_OUTPUT_DIR', '/tmp') + '/')
    parser.add_argument('--gpu', '-g', type=int, default=0)
    parser.add_argument('--cnn', action='store_true')
    parser.add_argument('--no_gnn', action='store_true')
    parser.add_argument('--use_wt', action='store_true')

    args = parser.parse_args()
    args.world_size = 1
    train(0, args)


def train(gpu, args):
    _ = torch.manual_seed(32)
    torch.cuda.set_device(gpu + args.gpu)
    device = torch.device('cuda:' + str(gpu + args.gpu))

    ## Grab metadata
    try:
        data_fpath = os.getenv('PT_DATA_DIR') + '/'
    except:
        home = str(pathlib.Path.home())
        data_fpath = home + '/data/'
    data_fpath += 'rocklin/'

    # grab model hyperparameters
    with open(args.config_fpath, 'r') as f:
        config = json.load(f)
    n_tokens = len(PROTEIN_ALPHABET)
    d_model = config['d_model']
    node_features = 10
    edge_features = 11
    dropout = config['dropout']
    use_mpnn = config['use_mpnn']
    n_layers = config['n_layers']
    n_connections = config['n_connections']

    # Prep results
    out_fname = 'rocklin.csv'
    if out_fname in os.listdir(args.out_fpath):
        record = pd.read_csv(args.out_fpath + out_fname)
    else:
        record = pd.read_csv(args.map_dir + 'metrics.csv', header=None)
        if args.no_gnn:
            record.columns = ['loss', 'accuracy', 'tokens', 'step']
            temp = pd.DataFrame(np.array([[np.nan, np.nan, 0, 0]]), columns=record.columns)
        else:
            record.columns = ['step', 'loss']
            record['epoch'] = record.index
            record['epoch'] += 1
            temp = pd.DataFrame(np.array([[0, np.nan, 0]]), columns=record.columns)
        record = pd.concat([record.iloc[np.arange(len(record) - 1, -1, -1)], temp], ignore_index=True)
        record = record.astype({'step': int})
    # Get the wt sequence, locations, and mutations
    df = pd.read_csv(data_fpath + 'rocklin.csv')
    pdb_ids = list(set(df['pdb_id']))

    # Things we need to process the structure
    collater = StructureCollater(SimpleCollater(PROTEIN_ALPHABET), n_connections=n_connections)
    tkn = Tokenizer(alphabet=PROTEIN_ALPHABET)
    pdbs = {}
    wts = {}
    for pdb_id in pdb_ids:
        coords, wt, _ = parse_PDB(data_fpath + pdb_id)
        wts[pdb_id] = (wt, torch.tensor(tkn.tokenize(wt)))

        # tokenized = torch.tensor(tkn.tokenize(wt))
        coords = {
            'N': coords[:, 0],
            'CA': coords[:, 1],
            'C': coords[:, 2]
        }
        dist, omega, theta, phi = process_coords(coords)
        batch = [[wt, torch.tensor(dist, dtype=torch.float),
                  torch.tensor(omega, dtype=torch.float),
                  torch.tensor(theta, dtype=torch.float), torch.tensor(phi, dtype=torch.float)]]
        _, nodes, edges, connections, edge_mask = collater(batch)
        nodes = nodes.to(device)
        edges = edges.to(device)
        connections = connections.to(device)
        edge_mask = edge_mask.to(device)
        pdbs[pdb_id] = (nodes, edges, connections, edge_mask)

    for idx, row in record.iterrows():
        step = str(int(row['step']))
        wd = args.map_dir + 'checkpoint' + step + '.tar'
        if args.cnn:
            d_cnn = 1280
            n_cnn_layers = 56
            kernel_size = 5
            r = 128
            slim = False
            d_embed = 8
            activation = 'gelu'
            causal = False
            pad_idx = PROTEIN_ALPHABET.index(PAD)
            cnn = ByteNetLM(n_tokens, d_embed, d_cnn, n_cnn_layers, kernel_size, r, final_ln=True,
                            slim=slim, activation=activation, causal=causal, padding_idx=pad_idx).to(device)

        if not args.no_gnn:
            gnn = BidirectionalStruct2SeqDecoder(n_tokens, node_features, edge_features,
                                               d_model, num_decoder_layers=n_layers,
                                               dropout=dropout, use_mpnn=use_mpnn,
                                                 pe=False, one_hot_src=(not args.cnn)).to(device)
        if int(row['step'] != 0):
            sd = torch.load(wd, map_location=device)
            if args.cnn and args.no_gnn:
                sd = sd['model_state_dict']
                sd = {k.split('module.')[1]: v for k, v in sd.items()}
                cnn.load_state_dict(sd)
                cnn = cnn.to(device).eval()
            elif args.cnn:
                gnn_sd = sd['decoder_state_dict']
                gnn.load_state_dict(gnn_sd)
                gnn = gnn.to(device).eval()
                cnn_sd = sd['encoder_state_dict']
                cnn_sd = {k.split('module.')[1]: v for k, v in cnn_sd.items()}
                cnn.load_state_dict(cnn_sd)
                cnn = cnn.to(device).eval()
            else:
                sd = sd['model_state_dict']
                sd = {k.split('module.')[1]: v for k, v in sd.items()}
                gnn.load_state_dict(sd)
                gnn = gnn.to(device).eval()

        pre = np.zeros(len(df))
        likelihoods = {pdb_id: {} for pdb_id in pdb_ids}
        for i, row in tqdm(df.iterrows(), total=len(df)):
            src = torch.tensor(tkn.tokenize(row['sequence'])).unsqueeze(0).to(device)
            pdb_id = row['pdb_id']
            nodes, edges, connections, edge_mask = pdbs[pdb_id]
            if args.use_wt:
                mut = row['sequence']
                wt, tokenized = wts[pdb_id]
                if mut == wt:
                    pre[i] = 0.0
                    continue
                # Get mutations
                aas = []
                pos = []
                for p, (s1, s2) in enumerate(zip(mut, wt)):
                    if s1 != s2:
                        aas.append(PROTEIN_ALPHABET.index(s1))
                        pos.append(p)
                # Compute likelihood if necessary
                pos = tuple(pos)
                if pos not in likelihoods[pdb_id]:
                    seq = ''
                    start = 0
                    for p in pos:
                        seq += wt[start:p]
                        start = p + 1
                        seq += MASK
                    seq += wt[start:]
                    src = torch.tensor(tkn.tokenize(seq)).unsqueeze(0).to(device)
                    with torch.no_grad():
                        if args.cnn:
                            src = cnn(src)
                            if args.no_gnn:
                                output = F.log_softmax(src, dim=-1)
                        if not args.no_gnn:
                            output = F.log_softmax(gnn(nodes, edges, connections, src, edge_mask), dim=-1)  # 1, ell, t
                        pl = output[0, pos].cpu()
                        wt_pl = output[torch.zeros(len(pos)).long(), pos, tokenized[list(pos)]].cpu()
                        likelihoods[pdb_id][pos] = (pl - wt_pl.unsqueeze(-1)).numpy()
                pre[i] = likelihoods[pdb_id][pos][np.arange(len(pos)), np.array(aas)].sum()
            else:
                with torch.no_grad():
                    if args.cnn:
                        src = cnn(src)
                        if args.no_gnn:
                            output = F.log_softmax(src, dim=-1)
                    if not args.no_gnn:
                        output = F.log_softmax(gnn(nodes, edges, connections, src, edge_mask), dim=-1) # 1, ell, t
                    # Slice out correct log probs
                    src = torch.tensor(tkn.tokenize(row['sequence'])).unsqueeze(0).to(device) # 1, ell
                    _, ell = src.shape
                    output = output[0, np.arange(ell), src[0]]
                    # Add them up
                    pre[i] = output.mean().cpu().item()
        df['p'] = pre
        rho = pearsonr(df['p'], df['fitness'])[0]
        record.loc[idx, 'overall_rho'] = rho
        print('overall', rho)
        rhos = 0.0
        for pdb_id in pdb_ids:
            d = df[df['pdb_id'] == pdb_id]
            rho = pearsonr(d['p'], d['fitness'])[0]
            print(pdb_id, rho)
            record.loc[idx, pdb_id[:-4]] = rho
            rhos += rho
        print('average', rhos / len(pdb_ids))
        record.to_csv(args.out_fpath + out_fname, index=False)

        # Save predictions
        df.to_csv(args.out_fpath + 'predictions%s.csv' %step, index=False)


if __name__ == '__main__':
    main()



