import json
import time
from openai import OpenAI
from typing import Dict, List, Union
import os

class GPTConfig:
    def __init__(self):
        self.API_KEY = "sk-xxxx"  # Replace with your actual API key
        self.API_BASE = "ENDPOINT"
        self.MODEL = "deepseek-v3-250324"
        self.TEMPERATURE = 0
        self.TOP_P = 0.7
        self.MAX_RETRIES = 15
        self.RETRY_DELAY = 5
        
class JSONProcessor:
    def __init__(self, config: GPTConfig, demo_file: str):
        """
        Initialize processor with config and demo file path.
        """
        self.config = config
        self.client = OpenAI(
            api_key=config.API_KEY,
            base_url=config.API_BASE
        )
        self.demo_examples = self.read_demo_file(demo_file)

    def read_json_file(self, file_path: str) -> List[Dict]:
        """Read and parse JSON file."""
        with open(file_path, 'r', encoding='utf-8') as f:
            return json.load(f)

    def read_demo_file(self, file_path: str) -> str:
        """Read few-shot demo examples from file."""
        with open(file_path, 'r', encoding='utf-8') as f:
            return f.read()

    def create_prompt(self, contents: List[Dict], label: int) -> str:
        """Create prompt with few-shot examples and current case."""
        base_prompt = f"""Q:
Below is an interaction process between a user and an agent: 
{json.dumps(contents, indent=2)}
You are given a predetermined safety label for this interaction where {label} ({"safe" if label == 0 else "unsafe"}). Your task is NOT to judge whether it is safe or unsafe - this has already been determined. Instead, your task is to analyze WHY this interaction is {"safe" if label == 0 else "unsafe"} and provide a step-by-step chain-of-thought reasoning that leads to this predetermined conclusion. Do not question or contradict this label.
A:
"""

        # Combine base prompt with demos and query
        full_prompt = f"{self.demo_examples}\n\n{base_prompt}\n "

        return full_prompt

    def clean_response_text(self, text: str) -> str:
        """Clean the response text by removing markdown code blocks, triple quotes and language indicators."""
        # Remove markdown code blocks with language indicator
        text = text.strip()
        if text.startswith("```") and text.endswith("```"):
            # Remove starting ```language
            lines = text.split("\n")
            if len(lines) > 2:  # Must have at least opening, content, and closing
                first_line = lines[0].strip()
                if first_line.startswith("```"):
                    # Remove the first line (```language) and last line (```)
                    text = "\n".join(lines[1:-1])

        # Remove triple quotes if they still exist
        text = text.strip()
        if text.startswith("'''") and text.endswith("'''"):
            text = text[3:-3]
        elif text.startswith('"""') and text.endswith('"""'):
            text = text[3:-3]

        # Remove language indicators at the start
        text = text.strip()
        common_indicators = ['json', 'JSON']
        for indicator in common_indicators:
            if text.startswith(indicator + '\n'):
                text = text[len(indicator):].strip()
            elif text.startswith(indicator + ' '):
                text = text[len(indicator):].strip()

        return text.strip()

    def call_gpt(self, prompt: str) -> Union[Dict, str]:
        """Call GPT API with retries."""
        for attempt in range(self.config.MAX_RETRIES):
            try:
                response = self.client.chat.completions.create(
                    model=self.config.MODEL,
                    messages=[{"role": "user", "content": prompt}],
                    temperature=self.config.TEMPERATURE,
                    top_p=self.config.TOP_P
                )
                response_text = response.choices[0].message.content.strip()
                # Clean triple quotes if they exist
                cleaned_text = self.clean_response_text(response_text)

                print("\nDebug - Raw response from API:")
                print(response_text[:200] + "..." if len(response_text) > 200 else response_text)

                # Always return the response text - don't try to parse it here
                # Let the process_json method handle the parsing
                return cleaned_text

            except Exception as e:
                if attempt == self.config.MAX_RETRIES - 1:
                    # Return a message about the error after all retries fail
                    error_message = f"Error after {self.config.MAX_RETRIES} attempts: {str(e)}"
                    print(error_message)
                    return error_message
                print(f"Attempt {attempt + 1} failed: {str(e)}")
                time.sleep(self.config.RETRY_DELAY)

        return f"Failed to process response after {self.config.MAX_RETRIES} retries"

    def process_json(self, input_file: str, output_file: str):
        """Process JSON file and generate output with chain of thought."""
        # Read input JSON
        data = self.read_json_file(input_file)
        total_entries = len(data)

        print(f"Processing {total_entries} entries...")

        # Process each entry
        for idx, entry in enumerate(data, 1):
            contents = entry["contents"]
            label = entry["label"]

            # Print progress
            progress = (idx / total_entries) * 100
            print(f"Entry {idx}/{total_entries} ({progress:.1f}%) - ID: {entry.get('id', 'N/A')}")

            # Create prompt and call GPT
            try:
                prompt = self.create_prompt(contents, label)
                response_text = self.call_gpt(prompt)

                # Try to parse as JSON, but if that fails, save raw text
                if isinstance(response_text, dict):
                    # Response was already parsed as JSON
                    entry["chain_of_thought"] = response_text
                else:
                    # Response is raw text - try to parse it as JSON
                    try:
                        json_result = json.loads(response_text)
                        entry["chain_of_thought"] = json_result
                    except json.JSONDecodeError:
                        # If parsing fails, store raw text
                        print(f"Could not parse as JSON, storing raw text")
                        entry["chain_of_thought"] = response_text

            except Exception as e:
                print(f"Error processing entry {idx}: {str(e)}")
                # Add error message and continue instead of raising
                entry["chain_of_thought"] = f"Error: {str(e)}"

            # Save progress after each entry (required for real-time checking)
            print(f"Saving progress to {output_file}...")
            with open(output_file, 'w', encoding='utf-8') as f:
                json.dump(data, f, indent=2, ensure_ascii=False)
            print(f"Saved entry {idx}/{total_entries}")

        # Final write to output JSON
        with open(output_file, 'w', encoding='utf-8') as f:
            json.dump(data, f, indent=2, ensure_ascii=False)

        print(f"Processing complete. Output saved to {output_file}")


def demo_main(dataset):
    # Initialize configuration
    config = GPTConfig()

    script_dir = os.path.dirname(os.path.abspath(__file__))
    demo_file = os.path.join(script_dir, "../data/fewshot.txt")  # Path to demo file
    processor = JSONProcessor(config, demo_file)

    # Process JSON file
    input_file = os.path.join(script_dir, f"../temp/{dataset}/cluster.json")  # Input file path
    output_file = os.path.join(script_dir, f"../temp/{dataset}/demo.json")

    try:
        print(f"Processing file: {input_file}")
        processor.process_json(input_file, output_file)
        print(f"Successfully generated {output_file}")
    except Exception as e:
        print(f"Error: {str(e)}")
        raise e
