import json
from typing import Tuple, List, Optional, Dict, Any
import os

def normalize_output(output_str: Any) -> int:
    output_str = str(output_str).strip().lower()

    if output_str in ['1', 'unsafe']:
        return 1
    elif output_str in ['0', 'safe']:
        return 0
    elif output_str == '1 (unsafe)':
        return 1
    elif output_str == '0 (safe)':
        return 0
    elif output_str.startswith('1'):
         # print(f"Warning: Ambiguous input '{output_str}', interpreting as 1.")
         return 1
    elif output_str.startswith('0'):
         # print(f"Warning: Ambiguous input '{output_str}', interpreting as 0.")
         return 0
    else:
        try:
            numeric_val = int(float(output_str))
            if numeric_val == 1:
                # print(f"Warning: Input '{output_str}' interpreted as numeric 1.")
                return 1
            elif numeric_val == 0:
                # print(f"Warning: Input '{output_str}' interpreted as numeric 0.")
                return 0
        except (ValueError, TypeError):
            raise ValueError(f"Unknown output format: {output_str}")
        raise ValueError(f"Numeric output format not binary (0 or 1): {output_str}")


def extract_output_from_chain_of_thought(chain_data: Dict[str, Any]) -> Optional[str]:
    output_keys = ['Output', 'Final Output', 'output', 'final_output', 'final output', 'Final Answer', 'final_answer']

    for key in output_keys:
        if key in chain_data:
            return str(chain_data[key])
    return None


def extract_output(output_data: Any) -> int:
    try:
        if isinstance(output_data, (str, int, float)):
             return normalize_output(output_data)

        if isinstance(output_data, dict):
            root_keys = ['Output', 'Final Output', 'output', 'final_output', 'final output', "final_safety_score", "prediction"]
            for key in root_keys:
                if key in output_data:
                    return normalize_output(output_data[key])

            if 'chain_of_thought' in output_data and isinstance(output_data['chain_of_thought'], dict):
                chain_data = output_data['chain_of_thought']
                output_value = extract_output_from_chain_of_thought(chain_data)
                if output_value is not None:
                    return normalize_output(output_value)

            if 'response' in output_data:
                 if isinstance(output_data['response'], str):
                     try:
                         return normalize_output(output_data['response'])
                     except ValueError:
                         pass

        raise ValueError(f"Could not find output value in expected format within: {output_data}")

    except ValueError as ve:
         raise ValueError(f"Error normalizing extracted output: {str(ve)} from data: {output_data}")
    except Exception as e:
        raise ValueError(f"Error processing output structure: {str(e)} in data: {output_data}")


def calculate_metrics(true_labels: List[int], predicted_labels: List[int]) -> Tuple[float, float, float, float]:
    if not true_labels:
         return 0.0, 0.0, 0.0, 0.0

    true_positives = sum(1 for t, p in zip(true_labels, predicted_labels) if t == 1 and p == 1)
    false_positives = sum(1 for t, p in zip(true_labels, predicted_labels) if t == 0 and p == 1)
    false_negatives = sum(1 for t, p in zip(true_labels, predicted_labels) if t == 1 and p == 0)
    true_negatives = sum(1 for t, p in zip(true_labels, predicted_labels) if t == 0 and p == 0)

    total_items = len(true_labels)
    accuracy = (true_positives + true_negatives) / total_items if total_items > 0 else 0.0

    precision = true_positives / (true_positives + false_positives) if (true_positives + false_positives) > 0 else 0.0

    recall = true_positives / (true_positives + false_negatives) if (true_positives + false_negatives) > 0 else 0.0

    f1 = 2 * (precision * recall) / (precision + recall) if (precision + recall) > 0 else 0.0

    return accuracy, precision, recall, f1


def process_json_file(file_path: str) -> None:
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            data = json.load(f)
    except FileNotFoundError:
        print(f"Error: File not found at {file_path}")
        return
    except json.JSONDecodeError:
        print(f"Error: Could not decode JSON from {file_path}")
        return
    except Exception as e:
        print(f"An unexpected error occurred opening the file: {e}")
        return

    if not isinstance(data, list):
        print(f"Error: Expected a JSON list, but got {type(data)}")
        return

    true_labels = []
    predicted_labels = []
    error_items = []
    total_items = len(data)

    print(f"Processing {total_items} items...")

    for index, item in enumerate(data):
        item_id = item.get('id', f"index_{index}")
        try:
            if not isinstance(item, dict):
                 raise TypeError(f"Item is not a dictionary: {item}")

            if 'label' not in item:
                 raise KeyError("Missing 'label' key")
            true_label = item['label']
            if true_label not in [0, 1]:
                 raise ValueError(f"Invalid true label value: {true_label}. Expected 0 or 1.")


            if 'output' not in item:
                 raise KeyError("Missing 'output' key")

            predicted_label = extract_output(item['output'])

            true_labels.append(true_label)
            predicted_labels.append(predicted_label)

        except (ValueError, KeyError, TypeError) as e:
            error_message = f"Error processing item {item_id}: {str(e)}"
            # print(error_message) # Suppress individual error prints for cleaner output if desired
            error_items.append({'id': item_id, 'error': str(e), 'item_data': item})
        except Exception as e:
             error_message = f"Unexpected error processing item {item_id}: {str(e)}"
             # print(error_message) # Suppress individual error prints
             error_items.append({'id': item_id, 'error': f"Unexpected: {str(e)}", 'item_data': item})


    print("\n=== Processing Summary ===")
    num_errors = len(error_items)
    num_success = len(true_labels)

    if error_items:
        print(f"\nItems with processing errors ({num_errors}):")
        print(f"IDs: {sorted([err['id'] for err in error_items])}")
        # with open("error_log.json", "w", encoding="utf-8") as err_f:
        #     json.dump(error_items, err_f, indent=4)
        # print("Detailed error information saved to error_log.json")


    if num_success > 0:
        accuracy, precision, recall, f1 = calculate_metrics(true_labels, predicted_labels)

        print(f"\nMetrics (calculated on {num_success} successfully processed items):")
        print(f"Accuracy:  {accuracy:.4f}")
        print(f"Precision: {precision:.4f}")
        print(f"Recall:    {recall:.4f}")
        print(f"F1 Score:  {f1:.4f}")

        print(f"\nSuccessfully processed: {num_success}/{total_items} items "
              f"({num_success / total_items * 100:.2f}%)")

        tp = sum(1 for t, p in zip(true_labels, predicted_labels) if t == 1 and p == 1)
        fp = sum(1 for t, p in zip(true_labels, predicted_labels) if t == 0 and p == 1)
        fn = sum(1 for t, p in zip(true_labels, predicted_labels) if t == 1 and p == 0)
        tn = sum(1 for t, p in zip(true_labels, predicted_labels) if t == 0 and p == 0)

        print("\nConfusion Matrix:")
        print(f"              Predicted 0   Predicted 1")
        print(f"Actual 0      {tn:<10}    {fp:<10}  (TN, FP)") 
        print(f"Actual 1      {fn:<10}    {tp:<10}  (FN, TP)")

    else:
        print("\nNo items were successfully processed to calculate metrics.")

def eval_main(dataset):
    script_dir = os.path.dirname(os.path.abspath(__file__))
    file_path = os.path.join(script_dir, f"../temp/{dataset}/output-k3_corrected.json")
    process_json_file(file_path)
