import json
# sys module removed as we don't need command line arguments
import re
from typing import Any, Dict, List, Tuple, Optional
import os

def extract_json_from_text(text: str) -> str:
    """
    Extract JSON content from a text that might contain both regular text and JSON.
    Looks for JSON content between ```json and ``` markers, or the last occurrence
    of a properly formatted JSON object.
    """
    # First try to find JSON between code block markers
    if "```json" in text:
        parts = text.split("```json")
        if len(parts) > 1:
            json_part = parts[1].split("```")[0]
            return json_part.strip()

    # If no code block markers, try to find the last occurrence of what looks like a JSON object
    import re
    # Find all sequences that look like JSON objects
    json_pattern = r'\{[^{}]*(?:\{[^{}]*\}[^{}]*)*\}'
    matches = list(re.finditer(json_pattern, text))

    if matches:
        # Return the last match
        return matches[-1].group()

    # If no JSON-like content found, return the original text
    return text


def clean_json_string(json_str: str) -> str:
    """
    Clean and prepare a JSON string for parsing by handling various formats
    and fixing common issues.
    """
    # Clean up the string
    json_str = json_str.strip()

    # First try to extract JSON content if mixed with regular text
    json_str = extract_json_from_text(json_str)

    # Remove Markdown code block markers if present
    if json_str.startswith('```json'):
        json_str = json_str[7:]
    if json_str.startswith('```'):
        json_str = json_str[3:]
    if json_str.endswith('```'):
        json_str = json_str[:-3]

    # If the string is wrapped in quotes and contains escaped quotes and newlines,
    # we need to first unescape it
    if json_str.startswith('"') and json_str.endswith('"'):
        # Use json.loads to properly unescape the string
        try:
            json_str = json.loads(json_str)
        except json.JSONDecodeError:
            # If this fails, continue with the original string
            pass

    # Handle double curly braces if present after unescaping
    if json_str.startswith('{{') and json_str.endswith('}}'):
        json_str = json_str[1:-1]  # Remove one set of braces

    # Find the last closing curly brace and trim anything after it
    try:
        last_brace_index = json_str.rindex('}')
        json_str = json_str[:last_brace_index + 1]
    except ValueError:
        pass  # No closing brace found

    # Strip any remaining whitespace
    json_str = json_str.strip()

    return json_str


def fix_string_concatenation(json_str: str) -> str:
    """
    Fix string concatenation in JSON by joining the strings.
    Handles patterns like: "string1" + "string2"
    """
    import re

    # Find all instances of string concatenation
    pattern = r'\"([^\"]*)\"\s*\+\s*\"([^\"]*)"'

    # Keep replacing until no more matches are found
    while True:
        match = re.search(pattern, json_str)
        if not match:
            break
        # Join the strings and replace the concatenation
        json_str = json_str[:match.start()] + f'"{match.group(1)}{match.group(2)}"' + json_str[match.end():]

    return json_str


def fix_json_string(json_str: str) -> Dict[str, Any]:
    """
    Fix and parse a JSON string that might be wrapped in code blocks, quotes,
    or have string concatenations.
    """
    # First clean the string
    json_str = clean_json_string(json_str)

    # Fix string concatenations if present
    json_str = fix_string_concatenation(json_str)

    # Parse the JSON
    return json.loads(json_str)


def fix_output_file(input_file: str, output_file: str = None):
    """
    Fix JSON formatting issues in the output file and remove 'original_fewshot_demos' keys.

    Args:
        input_file: Path to the input JSON file
        output_file: Path to save the fixed JSON file (default: input_file with '_fixed' suffix)
    """
    if output_file is None:
        # Create output filename by adding '_fixed' before the extension
        name_parts = input_file.rsplit('.', 1)
        output_file = f"{name_parts[0]}_fixed.{name_parts[1]}"

    try:
        # Read the input file
        print(f"Reading file: {input_file}")
        with open(input_file, 'r', encoding='utf-8') as f:
            data = json.load(f)

        fixed_count = 0
        error_count = 0
        removed_keys_count = 0

        # Process each item
        for item in data:
            # Remove 'original_fewshot_demos' if present
            if 'original_fewshot_demos' in item:
                del item['original_fewshot_demos']
                removed_keys_count += 1

            if 'combined_prompt' in item:
                del item['combined_prompt']
                removed_keys_count += 1

            if 'output' in item:
                try:
                    # Check if output is already a dictionary
                    if isinstance(item['output'], dict):
                        continue

                    # Try to parse the output string
                    item['output'] = fix_json_string(item['output'])

                    # Remove error field if it exists
                    if 'output_parse_error' in item:
                        del item['output_parse_error']

                    fixed_count += 1

                except Exception as e:
                    print(f"Warning: Could not fix output for item {item.get('id', 'unknown')}: {str(e)}")
                    error_count += 1

        # Save the fixed data
        print(f"\nSaving fixed file to: {output_file}")
        with open(output_file, 'w', encoding='utf-8') as f:
            json.dump(data, f, indent=2, ensure_ascii=False)

        print(f"\nProcessing complete!")
        print(f"Fixed items: {fixed_count}")
        print(f"Failed items: {error_count}")
        print(f"Removed 'original_fewshot_demos' keys: {removed_keys_count}")

        return output_file

    except FileNotFoundError:
        print(f"Error: Could not find input file: {input_file}")
        return None
    except json.JSONDecodeError as e:
        print(f"Error: Invalid JSON in input file: {str(e)}")
        return None
    except Exception as e:
        print(f"Error: An unexpected error occurred: {str(e)}")
        return None


def normalize_output(output_str: str) -> int:
    """Normalize different output formats to binary values."""
    # Remove any whitespace and convert to lowercase
    output_str = str(output_str).strip().lower()

    # Handle different formats
    if output_str in ['1', '1 (unsafe)', 'unsafe']:
        return 1
    elif output_str in ['0', '0 (safe)', 'safe']:
        return 0
    else:
        raise ValueError(f"Unknown output format: {output_str}")


def extract_output_from_chain_of_thought(chain_data: dict) -> Optional[str]:
    """Extract output from chain of thought dictionary, handling different output key names."""
    # Try different possible output key names
    output_keys = ['Output', 'Final Output', 'output', 'final_output', 'final output']

    for key in output_keys:
        if key in chain_data:
            return chain_data[key]
    return None


def extract_output(output_data: dict) -> int:
    """Extract output value from different output formats."""
    try:
        # Case 1: Direct output in chain_of_thought
        if isinstance(output_data, dict):
            if 'chain_of_thought' in output_data:
                chain_data = output_data['chain_of_thought']
                output_value = extract_output_from_chain_of_thought(chain_data)
                if output_value is not None:
                    return normalize_output(output_value)

            # Case 2: Direct output in the root
            for key in ['Output', 'Final Output', 'output', 'final_output', 'final output', "final_safety_score"]:
                if key in output_data:
                    return normalize_output(output_data[key])

        raise ValueError("Could not find output value in expected format")
    except Exception as e:
        raise ValueError(f"Error processing output: {str(e)}")


def calculate_metrics(true_labels: List[int], predicted_labels: List[int]) -> Tuple[float, float, float]:
    """Calculate Precision, Recall, and F1 score."""
    true_positives = sum(1 for t, p in zip(true_labels, predicted_labels) if t == 1 and p == 1)
    false_positives = sum(1 for t, p in zip(true_labels, predicted_labels) if t == 0 and p == 1)
    false_negatives = sum(1 for t, p in zip(true_labels, predicted_labels) if t == 1 and p == 0)

    precision = true_positives / (true_positives + false_positives) if (true_positives + false_positives) > 0 else 0
    recall = true_positives / (true_positives + false_negatives) if (true_positives + false_negatives) > 0 else 0
    f1 = 2 * (precision * recall) / (precision + recall) if (precision + recall) > 0 else 0

    return precision, recall, f1


def process_metrics(file_path: str) -> None:
    """Process the JSON file and calculate metrics."""
    with open(file_path, 'r', encoding='utf-8') as f:
        data = json.load(f)

    true_labels = []
    predicted_labels = []
    error_ids = []
    total_items = len(data)

    print(f"Processing {total_items} items for metrics calculation...")

    for item in data:
        try:
            # Extract the true label
            true_label = item['label']

            # Extract the predicted label from output
            predicted_label = extract_output(item['output'])

            true_labels.append(true_label)
            predicted_labels.append(predicted_label)

        except Exception as e:
            error_ids.append(item['id'])
            print(f"Error processing item {item['id']}: {str(e)}")

    # Print summary
    print("\n=== Processing Summary ===")
    if error_ids:
        print(f"\nItems with processing errors ({len(error_ids)}):")
        print(f"IDs: {sorted(error_ids)}")

    if true_labels and predicted_labels:
        precision, recall, f1 = calculate_metrics(true_labels, predicted_labels)
        print(f"\nMetrics:")
        print(f"Precision: {precision:.4f}")
        print(f"Recall: {recall:.4f}")
        print(f"F1 Score: {f1:.4f}")
        print(f"\nSuccessfully processed: {len(true_labels)}/{total_items} items "
              f"({len(true_labels) / total_items * 100:.2f}%)")
    else:
        print("No valid items to calculate metrics")

    # Print confusion matrix
    if true_labels and predicted_labels:
        tp = sum(1 for t, p in zip(true_labels, predicted_labels) if t == 1 and p == 1)
        fp = sum(1 for t, p in zip(true_labels, predicted_labels) if t == 0 and p == 1)
        fn = sum(1 for t, p in zip(true_labels, predicted_labels) if t == 1 and p == 0)
        tn = sum(1 for t, p in zip(true_labels, predicted_labels) if t == 0 and p == 0)

        print("\nConfusion Matrix:")
        print(f"True Positives: {tp}")
        print(f"False Positives: {fp}")
        print(f"False Negatives: {fn}")
        print(f"True Negatives: {tn}")


def fix1_main(dataset):
    script_dir = os.path.dirname(os.path.abspath(__file__))
    """Main function to run the combined script."""
    # Use the default file path from metrics script
    file_path = os.path.join(script_dir, f"../temp/{dataset}/output-k3.json")
    print(f"Using file: {file_path}")

    # Step 1: Fix the JSON file
    print("=== STEP 1: Fixing JSON File ===")
    fixed_file_path = fix_output_file(file_path)

    # Check if fixing was successful
    if fixed_file_path:
        # Step 2: Calculate metrics using the fixed file
        print("\n=== STEP 2: Calculating Metrics ===")
        process_metrics(fixed_file_path)
    else:
        print("Failed to fix the JSON file. Cannot proceed with metrics calculation.")
