import argparse
import time
import math
import numpy as np
import torch
import torch.nn as nn
import os
import data
import model
from optimizers import *
from torch.optim.lr_scheduler import ReduceLROnPlateau
from utils import batchify, get_batch, repackage_hidden

parser = argparse.ArgumentParser(description='PyTorch PennTreeBank RNN/LSTM Language Model')
parser.add_argument('--data', type=str, default='data/wikitext-2/',
                    help='location of the data corpus')
parser.add_argument('--model', type=str, default='LSTM',
                    help='type of recurrent net (LSTM, QRNN, GRU)')
parser.add_argument('--emsize', type=int, default=400,
                    help='size of word embeddings')
parser.add_argument('--nhid', type=int, default=1150,
                    help='number of hidden units per layer')
parser.add_argument('--nlayers', type=int, default=3,
                    help='number of layers')
parser.add_argument('--lr', type=float, default=30,
                    help='initial learning rate')
parser.add_argument('--clip', type=float, default=0.25,
                    help='gradient clipping')
parser.add_argument('--epochs', type=int, default=750,
                    help='upper epoch limit')
parser.add_argument('--batch_size', type=int, default=80, metavar='N',
                    help='batch size')
parser.add_argument('--bptt', type=int, default=70,
                    help='sequence length')
parser.add_argument('--dropout', type=float, default=0.4,
                    help='dropout applied to layers (0 = no dropout)')
parser.add_argument('--dropouth', type=float, default=0.2,
                    help='dropout for rnn layers (0 = no dropout)')
parser.add_argument('--dropouti', type=float, default=0.65,
                    help='dropout for input embedding layers (0 = no dropout)')
parser.add_argument('--dropoute', type=float, default=0.1,
                    help='dropout to remove words from embedding layer (0 = no dropout)')
parser.add_argument('--wdrop', type=float, default=0.5,
                    help='amount of weight dropout to apply to the RNN hidden to hidden matrix')
parser.add_argument('--seed', type=int, default=1882,
                    help='random seed')
parser.add_argument('--nonmono', type=int, default=5,
                    help='random seed')
parser.add_argument('--cuda', action='store_false',
                    help='use CUDA')
parser.add_argument('--log-interval', type=int, default=200, metavar='N',
                    help='report interval')
randomhash = ''.join(str(time.time()).split('.'))
parser.add_argument('--save', type=str,  default='wiki2',
                    help='path to save the final model')
parser.add_argument('--alpha', type=float, default=2,
                    help='alpha L2 regularization on RNN activation (alpha = 0 means no regularization)')
parser.add_argument('--beta', type=float, default=1,
                    help='beta slowness regularization applied on RNN activiation (beta = 0 means no regularization)')
parser.add_argument('--wdecay', type=float, default=1.2e-6,
                    help='weight decay applied to all weights')
parser.add_argument('--resume', type=str,  default='',
                    help='path of model to resume')
parser.add_argument('--when', nargs="+", type=int, default=[-1],
                    help='When (which epochs) to divide the learning rate by 10 - accepts multiple')
parser.add_argument('--run', type=int, default=0,
                    help='Number of runs')


parser.add_argument('--beta_theopoula', type=float, default=1e12,
                    help='beta for theopoula')
parser.add_argument('--beta1', type=float, default=0.9,
                    help='beta1 value')
parser.add_argument('--beta2', type=float, default=0.999,
                    help='bets2 value')
parser.add_argument('--eps', type=float,  default=1e-8)
parser.add_argument('--sync_eps', default=False)
parser.add_argument('--momentum', type=float, default=0.0)
parser.add_argument('--lr_gamma', default=0.1, type=float, help='learning rate')
parser.add_argument('--eps_gamma', default=1, type=float)
parser.add_argument('--final_lr', default=30, type=float, help='final learning rate of AdaBound')
parser.add_argument('--gamma', default=1e-3, type=float, help='convergence speed term of AdaBound')
parser.add_argument('--optimizer', type=str,  default='theopoula', help='optimizer to use (sgd, adam)')
parser.add_argument('--type_lrdecay', type=str, default='fixed_decay', choices = ['dev_decay', 'fixed_decay'])
parser.add_argument('--lr_patience', type=int, default=20)

args = parser.parse_args()
args.tied = True

# Set the random seed manually for reproducibility.
np.random.seed(args.seed)
torch.manual_seed(args.seed)
if torch.cuda.is_available():
    if not args.cuda:
        print("WARNING: You have a CUDA device, so you should probably run with --cuda")
    else:
        torch.cuda.manual_seed(args.seed)


name = {
    'sgd': 'lr{}-momentum{}-wdecay{}'.format(args.lr, args.momentum, args.wdecay),
    'adam': 'lr{}-betas{}-{}-wdecay{}-eps{}'.format(args.lr, args.beta1, args.beta2, args.wdecay, args.eps),
    'amsgrad': 'lr{}-betas{}-{}-wdecay{}-eps{}'.format(args.lr, args.beta1, args.beta2, args.wdecay, args.eps),
    'rmsprop': 'lr{}-wdecay{}'.format(args.lr, args.wdecay),
    'theopoula': 'lr{}-eps{}-wdecay{}-sync_eps{}-'.format(args.lr, args.eps, args.wdecay, args.sync_eps) + '-beta %.1e' % (args.beta_theopoula),
    'tuslac': 'lr{}-wdecay{}-lrgamma{}-epsgamma{}'.format(args.lr, args.wdecay, args.lr_gamma,
                                                                       args.eps_gamma) + '-beta %.1e' % (args.beta_theopoula),
    'adamp': 'lr{}-betas{}-{}-wdecay{}-eps{}'.format(args.lr, args.beta1, args.beta2, args.wdecay, args.eps),
    'adabelief': 'lr{}-betas{}-{}-eps{}-wdecay{}'.format(args.lr, args.beta1, args.beta2, args.eps, args.wdecay),
    'adabound': 'lr{}-betas{}-{}-final_lr{}-gamma{}-wdecay{}'.format(args.lr, args.beta1, args.beta2, args.final_lr, args.gamma, args.wdecay),
}[args.optimizer]

args.save = args.save + '-nlayers{}-'.format(args.nlayers) +'seed-{}-{}-'.format(args.seed, args.optimizer)  \
             + name + '-clip-{}'.format(args.clip) + '-when-{}'.format(args.when) \
             + '{}-patience{}-lrgamma{}'.format(args.type_lrdecay, args.lr_patience, args.lr_gamma)


if not os.path.exists('curve'):
    os.mkdir('curve')




###############################################################################
# Load data
###############################################################################

def model_save(fn):
    with open(os.path.join('ckpt', fn), 'wb') as f:
        torch.save([model, criterion, optimizer], f)

def model_load(fn):
    global model, criterion, optimizer
    with open(os.path.join('ckpt', fn), 'rb') as f:
        model, criterion, optimizer = torch.load(f)

import os
import hashlib
fn = 'corpus.{}.data'.format(hashlib.md5(args.data.encode()).hexdigest())
if os.path.exists(fn):
    print('Loading cached dataset...')
    corpus = torch.load(fn)
else:
    print('Producing dataset...')
    corpus = data.Corpus(args.data)
    torch.save(corpus, fn)

eval_batch_size = 10
test_batch_size = 1
train_data = batchify(corpus.train, args.batch_size, args)
val_data = batchify(corpus.valid, eval_batch_size, args)
test_data = batchify(corpus.test, test_batch_size, args)

###############################################################################
# Build the model
###############################################################################

from splitcross import SplitCrossEntropyLoss
criterion = None

ntokens = len(corpus.dictionary)
model = model.RNNModel(args.model, ntokens, args.emsize, args.nhid, args.nlayers, args.dropout, args.dropouth, args.dropouti, args.dropoute, args.wdrop, args.tied)
###
if args.resume:
    print('Resuming model ...')
    model_load(args.resume)
    optimizer.param_groups[0]['lr'] = args.lr
    model.dropouti, model.dropouth, model.dropout, args.dropoute = args.dropouti, args.dropouth, args.dropout, args.dropoute
    if args.wdrop:
        from weight_drop import WeightDrop
        for rnn in model.rnns:
            if type(rnn) == WeightDrop: rnn.dropout = args.wdrop
            elif rnn.zoneout > 0: rnn.zoneout = args.wdrop
###
if not criterion:
    splits = []
    if ntokens > 500000:
        # One Billion
        # This produces fairly even matrix mults for the buckets:
        # 0: 11723136, 1: 10854630, 2: 11270961, 3: 11219422
        splits = [4200, 35000, 180000]
    elif ntokens > 75000:
        # WikiText-103
        splits = [2800, 20000, 76000]
    print('Using', splits)
    criterion = SplitCrossEntropyLoss(args.emsize, splits=splits, verbose=False)
###
if args.cuda:
    model = model.cuda()
    criterion = criterion.cuda()
###
params = list(model.parameters()) + list(criterion.parameters())
total_params = sum(x.size()[0] * x.size()[1] if len(x.size()) > 1 else x.size()[0] for x in params if x.size())
print('Args:', args)
print('Model total parameters:', total_params)

###############################################################################
# Training code
###############################################################################

def evaluate(data_source, batch_size=10):
    # Turn on evaluation mode which disables dropout.
    model.eval()
    if args.model == 'QRNN': model.reset()
    total_loss = 0
    ntokens = len(corpus.dictionary)
    hidden = model.init_hidden(batch_size)
    for i in range(0, data_source.size(0) - 1, args.bptt):
        data, targets = get_batch(data_source, i, args, evaluation=True)
        output, hidden = model(data, hidden)
        total_loss += len(data) * criterion(model.decoder.weight, model.decoder.bias, output, targets).data
        hidden = repackage_hidden(hidden)
    return total_loss.item() / len(data_source)


def train():
    # Turn on training mode which enables dropout.
    if args.model == 'QRNN': model.reset()
    total_loss = 0
    start_time = time.time()
    ntokens = len(corpus.dictionary)
    hidden = model.init_hidden(args.batch_size)
    batch, i = 0, 0
    while i < train_data.size(0) - 1 - 1:
        bptt = args.bptt if np.random.random() < 0.95 else args.bptt / 2.
        # Prevent excessively small or negative sequence lengths
        seq_len = max(5, int(np.random.normal(bptt, 5)))
        # There's a very small chance that it could select a very long sequence length resulting in OOM
        # seq_len = min(seq_len, args.bptt + 10)

        lr2 = optimizer.param_groups[0]['lr']
        optimizer.param_groups[0]['lr'] = lr2 * seq_len / args.bptt
        model.train()
        data, targets = get_batch(train_data, i, args, seq_len=seq_len)

        # Starting each batch, we detach the hidden state from how it was previously produced.
        # If we didn't, the model would try backpropagating all the way to start of the dataset.
        hidden = repackage_hidden(hidden)
        optimizer.zero_grad()

        output, hidden, rnn_hs, dropped_rnn_hs = model(data, hidden, return_h=True)
        raw_loss = criterion(model.decoder.weight, model.decoder.bias, output, targets)

        loss = raw_loss
        # Activiation Regularization
        if args.alpha: loss = loss + sum(args.alpha * dropped_rnn_h.pow(2).mean() for dropped_rnn_h in dropped_rnn_hs[-1:])
        # Temporal Activation Regularization (slowness)
        if args.beta: loss = loss + sum(args.beta * (rnn_h[1:] - rnn_h[:-1]).pow(2).mean() for rnn_h in rnn_hs[-1:])
        loss.backward()

        # `clip_grad_norm` helps prevent the exploding gradient problem in RNNs / LSTMs.
        if args.clip: torch.nn.utils.clip_grad_norm_(params, args.clip)
        optimizer.step()

        total_loss += raw_loss.data
        optimizer.param_groups[0]['lr'] = lr2
        if batch % args.log_interval == 0 and batch > 0:
            cur_loss = total_loss.item() / args.log_interval
            elapsed = time.time() - start_time
            print('| epoch {:3d} | {:5d}/{:5d} batches | lr {:05.5f} | ms/batch {:5.2f} | '
                    'loss {:5.2f} | ppl {:8.2f} | bpc {:8.3f}'.format(
                epoch, batch, len(train_data) // args.bptt, optimizer.param_groups[0]['lr'],
                elapsed * 1000 / args.log_interval, cur_loss, math.exp(cur_loss), cur_loss / math.log(2)))
            total_loss = 0
            start_time = time.time()
        ###
        batch += 1
        i += seq_len

    return math.exp(cur_loss)

# Loop over epochs.
lr = args.lr
best_val_loss = []
stored_loss = 100000000

# At any point you can hit Ctrl + C to break out of training early.
try:
    optimizer = None
    # Ensure the optimizer is optimizing params, which includes both the model's weights as well as the criterion's weight (i.e. Adaptive Softmax)
    if args.optimizer == 'sgd':
        optimizer = torch.optim.SGD(params, lr=args.lr, momentum=args.momentum, weight_decay=args.wdecay)
    elif args.optimizer == 'adam':
        optimizer = torch.optim.Adam(params, lr=args.lr, weight_decay=args.wdecay)
    elif args.optimizer == 'amsgrad':
        optimizer = torch.optim.Adam(params, lr=args.lr, amsgrad=True, weight_decay=args.wdecay)
    elif args.optimizer == 'rmsprop':
        optimizer = torch.optim.RMSprop(params, lr=args.lr, weight_decay=args.wdecay)
    elif args.optimizer == 'adamp':
        optimizer = AdamP(params, lr=args.lr, weight_decay=args.wdecay)
    elif args.optimizer.lower() == 'adabelief':
        optimizer = AdaBelief(params, lr=args.lr, weight_decay=args.wdecay,
                              eps=args.eps, betas=(args.beta1, args.beta2))
    elif args.optimizer == 'adabound':
        optimizer = AdaBound(params, lr=args.lr, weight_decay=args.wdecay, final_lr=args.final_lr, gamma=args.gamma)
    elif args.optimizer == 'theopoula':
        optimizer = THEOPOULA(params, lr=args.lr, eps=args.eps, beta=args.beta_theopoula, weight_decay=args.wdecay, sync_eps=args.sync_eps)
    elif args.optimizer == 'tuslac':
        optimizer = TUSLAc(params, lr=args.lr, weight_decay=args.wdecay, beta=args.beta_theopoula)

    train_losses, val_losses, test_losses = [], [], []
    scheduler = ReduceLROnPlateau(optimizer, mode='min', factor=args.lr_gamma, patience=args.lr_patience)
    averaging = False

    for epoch in range(1, args.epochs+1):
        epoch_start_time = time.time()
        train_loss = train()
        train_losses.append(train_loss)


        if args.optimizer == 'adabelief':
            if optimizer.param_groups[0]['averaging']:
                averaging = True

        if ('t0' in optimizer.param_groups[0]) | averaging:
            tmp = {}
            for prm in model.parameters():
                if prm in optimizer.state.keys():
                    tmp[prm] = prm.data.detach()
                    prm.data = optimizer.state[prm]['ax'].detach()

            val_loss2 = evaluate(val_data)
            test_loss = evaluate(test_data, test_batch_size)
            print('-' * 89)
            print('| end of epoch {:3d} | time: {:5.2f}s | valid loss {:5.2f} | '
                  'valid ppl {:8.2f} | test ppl {:8.3f}'.format(
                epoch, (time.time() - epoch_start_time), val_loss2, math.exp(val_loss2), math.exp(test_loss)))

            if val_loss2 < stored_loss:
                model_save(args.save)
                print('Saving Averaged!')
                stored_loss = val_loss2

            for prm in model.parameters():
                if prm in tmp.keys():
                    prm.data = tmp[prm].detach()
                    prm.requires_grad = True
            del tmp

            val_losses.append(math.exp(val_loss2))
            test_losses.append(math.exp(test_loss))

        else:
            val_loss = evaluate(val_data, eval_batch_size)
            test_loss = evaluate(test_data, test_batch_size)
            print('-' * 89)
            print('| end of epoch {:3d} | time: {:5.2f}s | valid loss {:5.2f} | '
                  'valid ppl {:8.2f} | test ppl {:8.3f}'.format(
                epoch, (time.time() - epoch_start_time), val_loss, math.exp(val_loss), math.exp(test_loss)))
            print('-' * 89)

            if val_loss < stored_loss:
                model_save(args.save)
                print('Saving model (new best validation)')
                stored_loss = val_loss

            if args.optimizer == 'sgd' and 't0' not in optimizer.param_groups[0] and (len(best_val_loss)>args.nonmono and val_loss > min(best_val_loss[:-args.nonmono])):
                print('Switching to ASGD')
                optimizer = torch.optim.ASGD(model.parameters(), lr=args.lr, t0=0, lambd=0., weight_decay=args.wdecay)

            if args.optimizer == 'adabelief' and not optimizer.param_groups[0]['averaging'] and (len(best_val_loss)>args.nonmono and min(best_val_loss[-args.nonmono:]) > min(best_val_loss[:-args.nonmono])):
                print('Switching to Averaged AdaBelief')
                lr = optimizer.param_groups[0]['lr']
                optimizer = AdaBelief(params, lr=args.lr, weight_decay=args.wdecay, eps=args.eps, betas=(args.beta1, args.beta2), averaging=True)

            best_val_loss.append(val_loss)
            val_losses.append(math.exp(val_loss))
            test_losses.append(math.exp(test_loss))


            if args.type_lrdecay == 'fixed_decay':
                if epoch in args.when:
                    print('Saving model before learning rate decreased')
                    model_save('{}.e{}'.format(args.save, epoch))
                    print('Dividing learning rate by 10')
                    for param_group in optimizer.param_groups:
                        param_group['lr'] *= args.lr_gamma
                        if args.optimizer == 'theopoula':
                            param_group['eps'] *= args.eps_gamma
                        current_lr = param_group['lr']
                    print('curent lr - ', current_lr)
            elif args.type_lrdecay == 'dev_decay':
                scheduler.step(val_loss)
                current_lr = optimizer.param_groups[0]['lr']
                print('current_lr - ', current_lr)

        torch.save({'valid_loss': val_losses, 'test_loss': test_losses},
                       os.path.join('curve', args.save))

except KeyboardInterrupt:
    print('-' * 89)
    print('Exiting from training early')

# Load the best saved model.
model_load(args.save)

# Run on test data.
test_loss = evaluate(test_data, test_batch_size)
print('=' * 89)
print('| End of training | test loss {:5.2f} | test ppl {:8.2f} | test bpc {:8.3f}'.format(
    test_loss, math.exp(test_loss), test_loss / math.log(2)))
print('=' * 89)
