import cvxpy as cp
import numpy as np

def solve_mcc_cvxpy_l1l2(B, c, l, u, lambda_2, lamabda_1, solver=cp.ECOS, verbose:bool=False):
    """
    Solve the minimum-cost circulation problem with L2 and L1 regularization using CVXPY.
    https://www.cvxpy.org/tutorial/advanced/index.html#solver-options

    Parameters:
    - B: Constraint matrix
    - c: Cost vector
    - l: Lower bound vector
    - u: Upper bound vector
    - lambda_2: L2 regularization parameter
    - lambda_1: L1 regularization parameter

    Returns:
    - result: Result of the optimization
    """

    # Number of variables
    n = len(c)

    # Define the flow variable
    f = cp.Variable(n)

    # Define the objective function with L2 and L1 regularization
    objective = cp.Minimize(cp.matmul(c, f) + lambda_2 * cp.norm(f, 2)**2 + lamabda_1 * cp.norm(f, 1))

    # Define the equality constraint
    constraints = [B.T @ f == np.zeros(B.shape[1])]

    # Define the box constraints
    constraints += [l <= f, f <= u]

    # Solve the problem
    problem = cp.Problem(objective, constraints)
    result = problem.solve(solver=solver, verbose=verbose)

    return result, f.value


def solve_mcc_log_barrier_cvxpy(B, c, l, u, mu, solver=cp.ECOS, verbose: bool = False):
    """
    Solve the minimum-cost circulation problem with log-barrier terms using CVXPY.

    Parameters:
    - B: Constraint matrix
    - c: Cost vector
    - l: Lower bound vector
    - u: Upper bound vector
    - mu: Barrier parameter

    Returns:
    - result: Result of the optimization
    - optimal_flow: Optimal flow vector
    """
    
    # Number of variables
    n = len(c)
    
    # Define the flow variable
    f = cp.Variable(n)

    # Define the objective function with log-barrier terms
    objective = cp.Minimize(cp.matmul(c, f) - mu * cp.sum(cp.log(u - f) + cp.log(f - l)))

    # Define the equality constraint
    constraints = [B.T @ f == np.zeros(B.shape[1])]

    # Define the box constraints
    # constraints += [l <= f, f <= u]

    # Solve the problem
    problem = cp.Problem(objective, constraints)
    result = problem.solve(solver=solver, verbose=verbose)

    return result, f.value
