#    Copyright 2023 Haotian Liu
#
#    Licensed under the Apache License, Version 2.0 (the "License");
#    you may not use this file except in compliance with the License.
#    You may obtain a copy of the License at
#
#        http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS,
#    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#    See the License for the specific language governing permissions and
#    limitations under the License.


from typing import List, Optional, Tuple, Union

import torch
import torch.nn as nn

from transformers import AutoConfig, AutoModelForCausalLM, \
                         LlamaConfig, LlamaModel, LlamaForCausalLM

from transformers.modeling_outputs import CausalLMOutputWithPast
from transformers.generation.utils import GenerateOutput

from ..llava_arch import LlavaMetaModel, LlavaMetaForCausalLM


class LlavaConfig(LlamaConfig):
    model_type = "llava_llama"


class LlavaLlamaModel(LlavaMetaModel, LlamaModel):
    config_class = LlavaConfig

    def __init__(self, config: LlamaConfig):
        super(LlavaLlamaModel, self).__init__(config)


class LlavaLlamaForCausalLM(LlamaForCausalLM, LlavaMetaForCausalLM):
    config_class = LlavaConfig

    def __init__(self, config):
        super(LlamaForCausalLM, self).__init__(config)
        self.model = LlavaLlamaModel(config)
        self.pretraining_tp = config.pretraining_tp
        self.vocab_size = config.vocab_size
        self.lm_head = nn.Linear(config.hidden_size, config.vocab_size, bias=False)

        # Initialize weights and apply final processing
        self.post_init()

    def get_model(self):
        return self.model

    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        labels: Optional[torch.LongTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        images: Optional[torch.FloatTensor] = None,
        image_sizes: Optional[List[List[int]]] = None,
        return_dict: Optional[bool] = None,
        image_index:Optional[torch.LongTensor] = None,
        topk:Optional[torch.FloatTensor] = None,
        select_layer:Optional[torch.FloatTensor] = None,
    ) -> Union[Tuple, CausalLMOutputWithPast]:

        if inputs_embeds is None:
            (
                input_ids,
                position_ids,
                attention_mask,
                past_key_values,
                inputs_embeds,
                labels,
                split_size#iou时要取消注释
            ) = self.prepare_inputs_labels_for_multimodal(
                input_ids,
                position_ids,
                attention_mask,
                past_key_values,
                labels,
                images,
                image_sizes
            )
       
    
        return super().forward(
            input_ids=input_ids,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_values=past_key_values,
            inputs_embeds=inputs_embeds,
            labels=labels,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
            image_index=image_index,
            topk=topk,
            select_layer=select_layer
        )

    @torch.no_grad()
    def generate(
        self,
        inputs: Optional[torch.Tensor] = None,
        images: Optional[torch.Tensor] = None,
        image_sizes: Optional[torch.Tensor] = None,
        image_index:Optional[torch.LongTensor] = None,
        select_layer:Optional[torch.FloatTensor] = None,
        **kwargs,
    ) -> Union[GenerateOutput, torch.LongTensor]:
        position_ids = kwargs.pop("position_ids", None)
        attention_mask = kwargs.pop("attention_mask", None)
        if "inputs_embeds" in kwargs:
            raise NotImplementedError("`inputs_embeds` is not supported")
        

        if images is not None:
            (
                inputs,
                position_ids,
                attention_mask,
                _,
                inputs_embeds,
                _,
                split_size
            ) = self.prepare_inputs_labels_for_multimodal(
                inputs,
                position_ids,
                attention_mask,
                None,
                None,
                images,
                image_sizes=image_sizes
            )
        else:
            inputs_embeds = self.get_model().embed_tokens(inputs)

        start_layer,end_layer=5,16
        layer1=self.model.layers[:end_layer]
        image_index=torch.ones((inputs_embeds.shape[1]),device=inputs_embeds.device,dtype=torch.long)
        image_index[:split_size[0]]=0
        image_index[-split_size[1]:]=0
        image_features=inputs_embeds[:,image_index==1,:]
        patch_vit_embeds=torch.mean(image_features.view(-1,64,inputs_embeds.shape[-1]),dim=1)#image_features[0,:,:]#
        image_out,question_out,select_layer=[],[],[]
        hidden_states=torch.cat([patch_vit_embeds.unsqueeze(0),inputs_embeds[:,-split_size[1]:,:]],dim=1)
        batch_size, seq_length =  hidden_states.shape[:2]
        position_ids1 = torch.arange(
            0, seq_length, dtype=torch.long, device=inputs_embeds.device
        )
        position_ids1 = position_ids1.unsqueeze(0)
        #attention_mask=torch.ones ((batch_size,batch_size,seq_length,seq_length),dtype=torch.long,device=inputs_embeds.device)

        layer_count=0
        for layer in layer1:
            tmp_layer_outputs = layer(
                hidden_states,
                position_ids=position_ids1,
                past_key_value=None,
                output_attentions=True,
                use_cache=False,
            )
            hidden_states = tmp_layer_outputs[0]
            image_embedding=tmp_layer_outputs[0][:,:patch_vit_embeds.shape[0],:]
            question_embedding=tmp_layer_outputs[0][:,patch_vit_embeds.shape[0]:,:]
            image_out.append(image_embedding)
            question_out.append(question_embedding)    
            if layer_count==end_layer-1:
                a=PREP(image_out[start_layer:],question_out[start_layer:])
                select_layer.append(a+start_layer+1)
            layer_count+=1
        kwargs['select_layer']=torch.tensor(select_layer,dtype=torch.long, device=inputs_embeds.device)
        kwargs['image_index']=torch.tensor(image_index,dtype=torch.long, device=inputs_embeds.device)
        topk=0.11#0.33 0.22 0.11
        kwargs['topk'] =topk
        return super().generate(
            position_ids=position_ids,
            attention_mask=attention_mask,
            inputs_embeds=inputs_embeds,
            **kwargs
        )

    def prepare_inputs_for_generation(self, input_ids, past_key_values=None,
                                      inputs_embeds=None, **kwargs):
        images = kwargs.pop("images", None)
        image_sizes = kwargs.pop("image_sizes", None)
        inputs = super().prepare_inputs_for_generation(
            input_ids, past_key_values=past_key_values, inputs_embeds=inputs_embeds, **kwargs
        )
        if images is not None:
            inputs['images'] = images
        if image_sizes is not None:
            inputs['image_sizes'] = image_sizes
        return inputs

AutoConfig.register("llava_llama", LlavaConfig)
AutoModelForCausalLM.register(LlavaConfig, LlavaLlamaForCausalLM)
import torch.nn.functional as F
import numpy as np
from sklearn.cluster import AgglomerativeClustering
def PREP(image_hidden, text_hidden):
    # 初始化列表
    all_entropy, all_kl_div, all_best_token, all_cross_attention = [], [], [], []
    softmax0 = torch.nn.Softmax(dim=0)
    prev_cross_attention = None  # 存储前一层的注意力分布
    
    for i, (image, text) in enumerate(zip(image_hidden, text_hidden)):
        # 计算交叉注意力
        cross_attention = torch.matmul(image[0],text[0].T)#
        # 计算用于排序的注意力分布 (c2)
        cross_attention_norm= softmax0(cross_attention / torch.sqrt(torch.tensor(image.shape[-1])).item())
        cross_attention_norm=torch.mean(cross_attention_norm,dim=1).to(torch.float32)
        cross_attention_norm/=cross_attention_norm.sum()
        
        if i==0:
            prev_cross_attention =cross_attention_norm
            continue
        # 计算熵
        entropy = torch.sum(-cross_attention_norm * torch.log(cross_attention_norm + 1e-32))
       
        all_entropy.append(entropy.item())
        all_cross_attention.append(cross_attention_norm)
        
        # 计算KL散度 (与前一层的比较)
        if prev_cross_attention is not None:
            kl_div = torch.sum(
                prev_cross_attention * 
                (torch.log( prev_cross_attention+ 1e-32) - 
                 torch.log(cross_attention_norm + 1e-32))
            )
            
            all_kl_div.append(kl_div.item())
        else:
            all_kl_div.append(0)  # 第一层没有前一层的KL散度
        
        # 更新前一层的注意力分布
        prev_cross_attention = cross_attention_norm
        
        # 获取排序后的token索引
        arg_cross = torch.argsort(cross_attention_norm, descending=True).tolist()
        all_best_token.append(arg_cross)

    # 转换为tensor
    all_entropy = torch.tensor(all_entropy)
    all_kl_div = torch.tensor(all_kl_div)
    all_entropy=(all_entropy-all_entropy.min())/(all_entropy.max()-all_entropy.min()+1e-10)
    all_kl_div =(all_kl_div-all_kl_div.min())/(all_kl_div.max()-all_kl_div.min()+1e-10)
    # 使用权重参数平衡两者
    alpha = 0.5  # 熵的权重
    beta = 1-alpha   # KL散度的权重
    scores = alpha * all_entropy + beta * all_kl_div
    
    print("kl",all_kl_div,"entropy",all_entropy)
    sorted_indices = torch.argsort(scores, descending=True)  
    k =1
    print("当前选择第k大",k,scores)
    select_layer = sorted_indices[k-1]
    #select_layer = torch.argmax(scores)

    return select_layer.item()




def calculate_reduced_flops(K, k_star, r, L, M, D, H, m,S):

    N=r*L
    l_pre=calculate_per_layer_tflops(r,D,H)
    l_m=calculate_per_layer_tflops(N*m,D,H)
    l_total=calculate_per_layer_tflops(N,D,H)
    
    reduced_tokens=(K-k_star)*(l_total-l_m)-k_star*l_pre-N-H*N**2-H*N*M
    return reduced_tokens
def calculate_total_transformer_flops(K, N, D, H):
    layer_flops=calculate_per_layer_tflops(N,D,H)
    total_flops = K * layer_flops
    return total_flops


def calculate_cache_compression_ratio(k_star, m, K):
    compression_ratio = (k_star + (K - k_star)*m) / K
    return compression_ratio
def total_performance(K, k_star, r ,L, M,S,D,m):
    print(K, k_star, r ,L, M,S,D,m)
    H=28
    reduced_flops= calculate_reduced_flops(K, k_star, r, L, M, D, H, m,S) 
    total_flops=calculate_total_transformer_flops(K,r*L,D,H)
    cache_ratio=calculate_cache_compression_ratio(k_star,m,K)
    
    return float( reduced_flops),float(total_flops),cache_ratio

def calculate_per_layer_tflops(N, D, H):

    return 4 * N * (D ** 2) + 2 * (N ** 2) * D + (3 * N * (D ** 2)) / H