from typing import Optional
import os
import json
import logging
from pathlib import Path
from omegaconf import OmegaConf
from dataclasses import asdict, is_dataclass
from ..models.config_registry import CONFIG_REGISTRY

class AutoConfig:
    @classmethod
    def for_model(cls, model_name: str, preset: Optional[str] = None, **kwargs):
        """
        Create a configuration instance for the given model name and optional preset.

        Args:
            model_name (str): Name of the model (e.g., 'bert', 'roberta').
            preset (str, optional): Preset configuration (e.g., 'base', 'large').
            **kwargs: Additional configuration overrides.

        Returns:
            Config: Instantiated configuration object.

        Raises:
            ValueError: If model_name is not registered or preset is unsupported.
        """
        model_name = model_name.lower()
        if model_name not in cls.CONFIG_REGISTRY:
            raise ValueError(
                f"Unsupported model_name: '{model_name}'. "
                f"Available: {list(cls.CONFIG_REGISTRY.keys())}"
            )

        config_class = cls.CONFIG_REGISTRY[model_name]

        if preset:
            if not hasattr(config_class, "from_preset"):
                raise ValueError(
                    f"Model '{model_name}' does not support presets, but got preset='{preset}'."
                )
            config = config_class.from_preset(preset)
            if kwargs:
                for k, v in kwargs.items():
                    setattr(config, k, v)
            logging.info(f"[ConfigLoader] Created {config_class.__name__} for '{model_name}' with preset '{preset}' and overrides: {kwargs}")
        else:
            config = config_class(**kwargs)
            logging.info(f"[ConfigLoader] Created {config_class.__name__} for '{model_name}' with default values and overrides: {kwargs}")

        return config

    
    @classmethod
    def from_preset(cls, model_name: str, preset: Optional[str] = None, **kwargs):
        """
        Create a configuration for a specified model name and optional preset.

        Args:
            model_name (str): Name of model (e.g., 'bert', 'zipformer-tag').
            preset (str, optional): Preset configuration (e.g., 'base', 'large'). Defaults to None.
            **kwargs: Configuration parameters to override.

        Returns:
            Config: Instance of the appropriate configuration class.

        Raises:
            ValueError: If model_name or preset is not supported.
        """
        model_name = model_name.lower()
        if model_name not in CONFIG_REGISTRY:
            raise ValueError(
                f"Unsupported model_name: {model_name}. "
                f"Supported names: {list(CONFIG_REGISTRY.keys())}"
            )

        config_class = CONFIG_REGISTRY[model_name]
        if preset:
            if not hasattr(config_class, "from_preset"):
                raise ValueError(
                    f"Unsupported preset '{preset}' for model name '{model_name}'. "
                )
            config = config_class.from_preset(preset, **kwargs)
            logging.info(f"Initialized {config_class.__name__} for model name '{model_name}' with preset '{preset}'")
        else:
            config = config_class(**kwargs)
            logging.info(f"Initialized {config_class.__name__} for model name '{model_name}' with default values")
        return config
    
    @classmethod
    def from_pretrained(cls, pretrained_path: str, **kwargs):
        """
        Load a configuration from a pretrained directory.

        Args:
            pretrained_path (str): Path to directory with config.json.
            **kwargs: Configuration parameters to override.

        Returns:
            Config: Instance of the appropriate configuration class.

        Raises:
            FileNotFoundError: If config.json is not found.
            ValueError: If 'name' key is missing in config.json.
        """
        config_file = os.path.join(pretrained_path, "config.json")
        if not (os.path.isdir(pretrained_path) and os.path.exists(config_file)):
            raise FileNotFoundError(f"No config.json found in {pretrained_path}")

        with open(config_file, "r") as f:
            config_dict = json.load(f)

        config_dict.update(kwargs)
        model_name = config_dict["name"].lower()
        return cls.from_preset(model_name, preset=None, **config_dict)

    @classmethod
    def from_hydra_cfg(cls, hydra_cfg):
        if not hasattr(hydra_cfg, "name"):
            raise ValueError("hydra_cfg must contain 'name' key")

        model_name = hydra_cfg.name.lower()
        if model_name not in CONFIG_REGISTRY:
            raise ValueError(
                f"Unsupported model name '{model_name}'. "
                f"Supported names: {list(CONFIG_REGISTRY.keys())}"
            )

        # Check mutual exclusivity
        has_config_path = bool(hydra_cfg.get("config_path"))
        has_preset = bool(hydra_cfg.get("config_preset"))
        if has_config_path and has_preset:
            raise ValueError("hydra_cfg cannot contain both 'config_preset' and 'config_path'")

        # Extract override values (may be empty dict)
        if hydra_cfg.get("config") is None:
            overrides = {}
        else:
            overrides = OmegaConf.to_container(hydra_cfg.config, resolve=True)

        if has_config_path:
            config_path = hydra_cfg.config_path
            config = cls.from_pretrained(config_path, **overrides)
            logging.info(f"Created config for model '{model_name}' from config_path '{config_path}' with overrides")
            return config

        # Use preset (even if None)
        preset = hydra_cfg.get("config_preset", None)
        config = cls.from_preset(model_name, preset=preset, **overrides)
        logging.info(f"Created config for model '{model_name}' with preset '{preset or 'default'}'")
        return config

