import base64
import gzip
from typing import Iterable, Optional
from argparse import Namespace

import k2
import numpy as np
import torch
import torch.nn.functional as F
from torch import Tensor, nn

from .decoding import decode as decode_function
from .decoding import detect_language as detect_language_function
from .model_config import WhisperConfig
from .label_smoothing import LabelSmoothingLoss
from .load import load_model
from ..zipformer.utils.padding import make_pad_mask


class LayerNorm(nn.LayerNorm):
    def forward(self, x: Tensor) -> Tensor:
        return super().forward(x.float()).type(x.dtype)


class Linear(nn.Linear):
    def forward(self, x: Tensor) -> Tensor:
        return F.linear(
            x,
            self.weight.to(x.dtype),
            None if self.bias is None else self.bias.to(x.dtype),
        )


class Conv1d(nn.Conv1d):
    def _conv_forward(
        self, x: Tensor, weight: Tensor, bias: Optional[Tensor]
    ) -> Tensor:
        return super()._conv_forward(
            x, weight.to(x.dtype), None if bias is None else bias.to(x.dtype)
        )


def sinusoids(length, channels, max_timescale=10000):
    """Returns sinusoids for positional embedding"""
    assert channels % 2 == 0
    log_timescale_increment = np.log(max_timescale) / (channels // 2 - 1)
    inv_timescales = torch.exp(-log_timescale_increment * torch.arange(channels // 2))
    scaled_time = torch.arange(length)[:, np.newaxis] * inv_timescales[np.newaxis, :]
    return torch.cat([torch.sin(scaled_time), torch.cos(scaled_time)], dim=1)


class MultiHeadAttention(nn.Module):
    def __init__(self, n_state: int, n_head: int):
        super().__init__()
        self.n_head = n_head
        self.query = Linear(n_state, n_state)
        self.key = Linear(n_state, n_state, bias=False)
        self.value = Linear(n_state, n_state)
        self.out = Linear(n_state, n_state)

    def forward(
        self,
        x: Tensor,
        xa: Optional[Tensor] = None,
        mask: Optional[Tensor] = None,
        kv_cache: Optional[dict] = None,
    ):
        q = self.query(x)

        if kv_cache is None or xa is None or self.key not in kv_cache:
            # hooks, if installed (i.e. kv_cache is not None), will prepend the cached kv tensors;
            # otherwise, perform key/value projections for self- or cross-attention as usual.
            k = self.key(x if xa is None else xa)
            v = self.value(x if xa is None else xa)
        else:
            # for cross-attention, calculate keys and values once and reuse in subsequent calls.
            k = kv_cache[self.key]
            v = kv_cache[self.value]

        wv, qk = self.qkv_attention(q, k, v, mask)
        return self.out(wv), qk

    def qkv_attention(
        self, q: Tensor, k: Tensor, v: Tensor, mask: Optional[Tensor] = None
    ):
        n_batch, n_ctx, n_state = q.shape
        scale = (n_state // self.n_head) ** -0.25
        q = q.view(*q.shape[:2], self.n_head, -1).permute(0, 2, 1, 3) * scale
        k = k.view(*k.shape[:2], self.n_head, -1).permute(0, 2, 3, 1) * scale
        v = v.view(*v.shape[:2], self.n_head, -1).permute(0, 2, 1, 3)

        qk = q @ k
        if mask is not None:
            qk = qk + mask[:n_ctx, :n_ctx]
        qk = qk.float()

        w = F.softmax(qk, dim=-1).to(q.dtype)
        return (w @ v).permute(0, 2, 1, 3).flatten(start_dim=2), qk.detach()


class ResidualAttentionBlock(nn.Module):
    def __init__(self, n_state: int, n_head: int, cross_attention: bool = False):
        super().__init__()

        self.attn = MultiHeadAttention(n_state, n_head)
        self.attn_ln = LayerNorm(n_state)

        self.cross_attn = (
            MultiHeadAttention(n_state, n_head) if cross_attention else None
        )
        self.cross_attn_ln = LayerNorm(n_state) if cross_attention else None

        n_mlp = n_state * 4
        self.mlp = nn.Sequential(
            Linear(n_state, n_mlp), nn.GELU(), Linear(n_mlp, n_state)
        )
        self.mlp_ln = LayerNorm(n_state)

    def forward(
        self,
        x: Tensor,
        xa: Optional[Tensor] = None,
        mask: Optional[Tensor] = None,
        kv_cache: Optional[dict] = None,
    ):
        x = x + self.attn(self.attn_ln(x), mask=mask, kv_cache=kv_cache)[0]
        if self.cross_attn:
            x = x + self.cross_attn(self.cross_attn_ln(x), xa, kv_cache=kv_cache)[0]
        x = x + self.mlp(self.mlp_ln(x))
        return x


class AudioEncoder(nn.Module):
    def __init__(
        self, n_mels: int, n_ctx: int, n_state: int, n_head: int, n_layer: int
    ):
        super().__init__()
        self.conv1 = Conv1d(n_mels, n_state, kernel_size=3, padding=1)
        self.conv2 = Conv1d(n_state, n_state, kernel_size=3, stride=2, padding=1)
        self.register_buffer("positional_embedding", sinusoids(n_ctx, n_state))

        self.blocks: Iterable[ResidualAttentionBlock] = nn.ModuleList(
            [ResidualAttentionBlock(n_state, n_head) for _ in range(n_layer)]
        )
        self.ln_post = LayerNorm(n_state)

    def forward(self, x: Tensor):
        """
        x : torch.Tensor, shape = (batch_size, n_mels, n_ctx)
            the mel spectrogram of the audio
        """
        x = F.gelu(self.conv1(x))
        x = F.gelu(self.conv2(x))
        x = x.permute(0, 2, 1)

        x = (x + self.positional_embedding[:x.shape[1]]).to(x.dtype)

        for block in self.blocks:
            x = block(x)

        x = self.ln_post(x)
        return x


class TextDecoder(nn.Module):
    def __init__(
        self, n_vocab: int, n_ctx: int, n_state: int, n_head: int, n_layer: int
    ):
        super().__init__()

        self.token_embedding = nn.Embedding(n_vocab, n_state)
        self.positional_embedding = nn.Parameter(torch.empty(n_ctx, n_state))

        self.blocks: Iterable[ResidualAttentionBlock] = nn.ModuleList(
            [
                ResidualAttentionBlock(n_state, n_head, cross_attention=True)
                for _ in range(n_layer)
            ]
        )
        self.ln = LayerNorm(n_state)

        mask = torch.empty(n_ctx, n_ctx).fill_(-np.inf).triu_(1)
        self.register_buffer("mask", mask, persistent=False)

    def forward(self, x: Tensor, xa: Tensor, kv_cache: Optional[dict] = None):
        """
        x : torch.LongTensor, shape = (batch_size, <= n_ctx)
            the text tokens
        xa : torch.Tensor, shape = (batch_size, n_audio_ctx, n_audio_state)
            the encoded audio features to be attended on
        """
        offset = next(iter(kv_cache.values())).shape[1] if kv_cache else 0
        x = (
            self.token_embedding(x)
            + self.positional_embedding[offset : offset + x.shape[-1]]
        )
        x = x.to(xa.dtype)

        for block in self.blocks:
            x = block(x, xa, mask=self.mask, kv_cache=kv_cache)

        x = self.ln(x)
        logits = (
            x @ torch.transpose(self.token_embedding.weight.to(x.dtype), 0, 1)
        ).float()

        return logits


class WhisperAsrModel(nn.Module):
    def __init__(self, config: WhisperConfig):
        super().__init__()
        self.config = config
        self.dims = config.dims
        self.apply_padding_mask = config.apply_padding_mask
        self.pad = config.blank_id
        self.is_multilingual = config.is_multilingual

        self.encoder = AudioEncoder(
            self.dims.n_mels,
            self.dims.n_audio_ctx,
            self.dims.n_audio_state,
            self.dims.n_audio_head,
            self.dims.n_audio_layer,
        )
        self.decoder = TextDecoder(
            self.dims.n_vocab,
            self.dims.n_text_ctx,
            self.dims.n_text_state,
            self.dims.n_text_head,
            self.dims.n_text_layer,
        )
        
        # TODO(jiahong): disable alignment_heads for cosnistency with DDP,
        #                need a better solution
        # # use the last half among the decoder layers for time alignment by default;
        # # to use a specific set of heads, see `set_alignment_heads()` below.
        # all_heads = torch.zeros(
        #     self.dims.n_text_layer, self.dims.n_text_head, dtype=torch.bool
        # )
        # all_heads[self.dims.n_text_layer // 2 :] = True
        # self.register_buffer("alignment_heads", all_heads.to_sparse(), persistent=False)

        # NOTE(jiahong): params in token_embedding can be wrong due to extra LID/tokens
        self.load_state_dict(load_model(config.model_name), strict=False)
        self.rearange_token_embedding()

    def rearange_token_embedding(self):
        """ Before: text_tokens + lang_tokens     + task_tokens(6)     + time_tokens(1501)
            After : text_tokens + lang_tokens + L + task_tokens(6) + C + time_tokens(1501)
        """
        # 1) copy weights to a new embedding table
        embedding_old = self.decoder.token_embedding
        embedding = torch.nn.Embedding(self.config.vocab_size, embedding_old.weight.size(1))
        embedding.to(embedding_old.weight.device, dtype=embedding_old.weight.dtype)
        torch.nn.init.xavier_uniform_(embedding.weight, gain=0.001)
        embedding.weight.data[: embedding_old.weight.size(0)] = embedding_old.weight.data

        # 2) rearange token embeddings
        L, C = len(self.config.extra_languages), len(self.config.extra_tokens)
        index1 = self.config.vocab_size - C - 1507

        if self.config.model_name == "large-v3":
            index2 = index1 - L + 1 
            assert next(iter(self.config.extra_languages)) == 'yue', \
                "<yue> should always be given as the first extra language in large-v3"
            del self.config.extra_languages['yue']
        else:
            index2 = index1 - L
        embedding.weight.data[index1 + 6 + C :] = embedding_old.weight.data[index2 + 6 : index2 + 1507]
        embedding.weight.data[index1 : index1 + 6] = embedding_old.weight.data[index2 : index2 + 6]

        del embedding_old
        self.decoder.token_embedding = embedding

    def set_alignment_heads(self, dump: bytes):
        array = np.frombuffer(
            gzip.decompress(base64.b85decode(dump)), dtype=bool
        ).copy()
        mask = torch.from_numpy(array).reshape(
            self.dims.n_text_layer, self.dims.n_text_head
        )
        self.register_buffer("alignment_heads", mask.to_sparse(), persistent=False)

    def embed_audio(self, mel: torch.Tensor):
        return self.encoder(mel)

    def logits(self, tokens: torch.Tensor, audio_features: torch.Tensor):
        return self.decoder(tokens, audio_features)

    @property
    def device(self):
        return next(self.parameters()).device

    def install_kv_cache_hooks(self, cache: Optional[dict] = None):
        """
        The `MultiHeadAttention` module optionally accepts `kv_cache` which stores the key and value
        tensors calculated for the previous positions. This method returns a dictionary that stores
        all caches, and the necessary hooks for the key and value projection modules that save the
        intermediate tensors to be reused during later calculations.

        Returns
        -------
        cache : Dict[nn.Module, torch.Tensor]
            A dictionary object mapping the key/value projection modules to its cache
        hooks : List[RemovableHandle]
            List of PyTorch RemovableHandle objects to stop the hooks to be called
        """
        cache = {**cache} if cache is not None else {}
        hooks = []

        def save_to_cache(module, _, output):
            if module not in cache or output.shape[1] > self.dims.n_text_ctx:
                # save as-is, for the first token or cross attention
                cache[module] = output
            else:
                cache[module] = torch.cat([cache[module], output], dim=1).detach()
            return cache[module]

        def install_hooks(layer: nn.Module):
            if isinstance(layer, MultiHeadAttention):
                hooks.append(layer.key.register_forward_hook(save_to_cache))
                hooks.append(layer.value.register_forward_hook(save_to_cache))

        self.decoder.apply(install_hooks)
        return cache, hooks

    def forward_encoder(
        self, x: Tensor, x_lens: Optional[Tensor] = None
    ):
        """Compute encoder outputs.
        Args:
          x:
            A 3-D tensor of shape (N, T, C).
          x_lens:
            A 1-D tensor of shape (N,). It contains the number of frames in `x`
            before padding.

        Returns:
          encoder_out:
            Encoder output, of shape (N, T, C).
          encoder_out_lens:
            Encoder output lengths, of shape (N,).
        """
        if self.apply_padding_mask:
            encoder_out, encoder_out_lens = self.encoder(x, x_lens)
        else:
            encoder_out = self.encoder(x)
            encoder_out_lens = None
        
        return Namespace(
            encoder_out=encoder_out,
            encoder_out_lens=encoder_out_lens
        )

    def forward_decoder(
        self, 
        ys_pad: Tensor, 
        xa: Tensor, 
        xa_lens: Optional[Tensor] = None
    ):
        """Compute decoder outputs.
        Args:
          ys_pad: 
            A 3-D tensor of shape (N, L, C).
          xa:
            A 3-D tensor of shape (N, T, C).
          xa_lens:
            A 1-D tensor of shape (N,). It contains the number of frames in `xa` 
            before padding.

        Returns:
          decoder_out:
            decoder output, of shape (N, L, V).
        """
        if self.apply_padding_mask:
            xa_mask = make_pad_mask(xa_lens).unsqueeze(1)
            xa_mask = xa_mask.expand(-1, ys_pad.size(1), -1) * (-np.inf)
            decoder_output = self.decoder(ys_pad, xa, xa_mask)
        else:
            decoder_output = self.decoder(ys_pad, xa)

        return decoder_output
        
    def forward(
        self, 
        x: Tensor, 
        x_lens: Tensor, 
        y: k2.RaggedTensor, 
        target: Optional[k2.RaggedTensor] = None
    ) -> Tensor:
        """
        Args:
          x:
            A 3-D tensor of shape (N, T, C).
          x_lens:
            A 1-D tensor of shape (N,). It contains the number of frames in `x`
            before padding.
          y:
            A ragged tensor with 2 axes [utt][label]. It contains labels of each
            utterance.
        Returns:
          Return the attention decoder loss,
          in form of attention_decoder_loss
        """
        assert x.ndim == 3, x.shape
        assert x_lens.ndim == 1, x_lens.shape
        assert y.num_axes == 2, y.num_axes

        assert x.size(0) == x_lens.size(0) == y.dim0, (x.shape, x_lens.shape, y.dim0)

        device = x.device

        # Compute encoder outputs
        encoder_output = self.forward_encoder(x, x_lens)
        encoder_out = encoder_output.encoder_out
        encoder_out_lens = encoder_output.encoder_out_lens

        row_splits = y.shape.row_splits(1)
        y_lens = row_splits[1:] - row_splits[:-1]
        y_pad = y.pad(mode="constant", padding_value=self.pad)
        target = target.pad(mode="constant", padding_value=self.pad)

        attention_decoder_loss = self.calc_att_loss(
            encoder_out=encoder_out,
            encoder_out_lens=encoder_out_lens,
            y=y_pad,
            target=target
        )

        return attention_decoder_loss

    def calc_att_loss(
        self, 
        encoder_out: Tensor, 
        encoder_out_lens: Tensor, 
        y: k2.RaggedTensor, 
        target: k2.RaggedTensor
    ) -> Tensor:
        decoder_output = self.forward_decoder(y, encoder_out, encoder_out_lens)

        decoder_criterion = LabelSmoothingLoss(
            ignore_index=self.pad, label_smoothing=0.1, reduction="sum"
        )
        decoder_attention_loss = decoder_criterion(decoder_output, target)
        return decoder_attention_loss

    detect_language = detect_language_function
    decode = decode_function
