import json
import os
import random
import sys
from ast import literal_eval

import numpy as np
import torch


def set_seed(seed):
    random.seed(seed)
    np.random.seed(seed)
    torch.manual_seed(seed)
    torch.cuda.manual_seed_all(seed)


class CfgNode:
    

    def __init__(self, **kwargs):
        self.__dict__.update(kwargs)

    def __str__(self):
        return self._str_helper(0)

    def _str_helper(self, indent):
        
        parts = []
        for k, v in self.__dict__.items():
            if isinstance(v, CfgNode):
                parts.append("%s:\n" % k)
                parts.append(v._str_helper(indent + 1))
            else:
                parts.append("%s: %s\n" % (k, v))
        parts = [" " * (indent * 4) + p for p in parts]
        return "".join(parts)

    def to_dict(self):
        
        return {k: v.to_dict() if isinstance(v, CfgNode) else v for k, v in self.__dict__.items()}

    def merge_from_dict(self, d):
        self.__dict__.update(d)

    def merge_from_args(self, args):
        
        for arg in args:
            keyval = arg.split("=")
            assert len(keyval) == 2, "expecting each override arg to be of form --arg=value, got %s" % arg
            key, val = keyval  

            
            try:
                val = literal_eval(val)
                
            except ValueError:
                pass

            
            assert key[:2] == "--"
            key = key[2:]  
            keys = key.split(".")
            obj = self
            for k in keys[:-1]:
                obj = getattr(obj, k)
            leaf_key = keys[-1]

            
            assert hasattr(obj, leaf_key), f"{key} is not an attribute that exists in the config"

            
            print0("command line overwriting config attribute %s with %s" % (key, val))
            setattr(obj, leaf_key, val)


def print0(*args, **kwargs):
    if int(os.environ["RANK"]) == 0:
        print(*args, **kwargs)
