from collections import OrderedDict

import numpy as np
import torch
import torch.optim as optim
from torch import nn as nn
import torch.nn.functional as F

import rlkit.torch.pytorch_util as ptu
from rlkit.core.eval_util import create_stats_ordered_dict
from rlkit.torch.torch_rl_algorithm import TorchTrainer
from rlkit.samplers.data_collector.path_collector import MdpPathCollector

# from torch import autograd

class CQLTrainer(TorchTrainer):
    def __init__(
            self, 
            env,
            exp_name,
            policy,
            qf1,
            qf2,
            target_qf1,
            target_qf2,
            behavior_poliy,

            discount=0.99,
            reward_scale=1.0,

            policy_lr=1e-3,
            qf_lr=1e-3,
            optimizer_class=optim.Adam,

            soft_target_tau=1e-2,
            plotter=None,
            render_eval_paths=False,

            use_automatic_entropy_tuning=True,
            target_entropy=None,
            policy_eval_start=0,
            num_qs=2,

            # CQL
            min_q_version=3,
            temp=1.0,
            min_q_weight=1.0,
            mix_weight=0.7,

            ## sort of backup
            max_q_backup=False,
            deterministic_backup=True,
            num_total=50,
            with_lagrange=False,
            lagrange_thresh=0.0,
    ):
        super().__init__()
        self.env = env
        self.exp_name = exp_name
        self.policy = policy
        self.behavior_poliy = behavior_poliy
        self.qf1 = qf1
        self.qf2 = qf2
        self.target_qf1 = target_qf1
        self.target_qf2 = target_qf2
        self.soft_target_tau = soft_target_tau

        self.use_automatic_entropy_tuning = use_automatic_entropy_tuning
        if self.use_automatic_entropy_tuning:
            if target_entropy:
                self.target_entropy = target_entropy
            else:
                self.target_entropy = -np.prod(self.env.action_space.shape).item() 
            self.log_alpha = ptu.zeros(1, requires_grad=True)
            self.alpha_optimizer = optimizer_class(
                [self.log_alpha],
                lr=policy_lr,
            )
        
        self.with_lagrange = with_lagrange
        print("self.with_lagrange: ", self.with_lagrange)
        if self.with_lagrange:
            self.target_action_gap = lagrange_thresh
            print("lagrange_thresh: ", lagrange_thresh)
            self.log_alpha_prime = ptu.zeros(1, requires_grad=True)
            self.alpha_prime_optimizer = optimizer_class(
                [self.log_alpha_prime],
                lr=qf_lr,
            )

        self.plotter = plotter
        self.render_eval_paths = render_eval_paths

        self.qf_criterion = nn.MSELoss()
        self.vf_criterion = nn.MSELoss()

        self.policy_optimizer = optimizer_class(
            self.policy.parameters(),
            lr=policy_lr,
        )
        self.qf1_optimizer = optimizer_class(
            self.qf1.parameters(),
            lr=qf_lr,
        )
        self.qf2_optimizer = optimizer_class(
            self.qf2.parameters(),
            lr=qf_lr,
        )

        self.discount = discount
        self.reward_scale = reward_scale
        self.eval_statistics = OrderedDict()
        self.eval_wandb = OrderedDict()
        self._n_train_steps_total = 0
        self._need_to_update_eval_statistics = True
        self.policy_eval_start = policy_eval_start
        
        self._current_epoch = 0
        self._policy_update_ctr = 0
        self._num_q_update_steps = 0
        self._num_policy_update_steps = 0
        self._num_policy_steps = 1
        
        self.num_qs = num_qs

        ## min Q
        self.temp = temp
        self.min_q_version = min_q_version
        self.min_q_weight = min_q_weight

        print("self.min_q_version: \t", min_q_version)
        print("self.min_q_weight: \t", min_q_weight)
        print('self.mix_weight: \t', mix_weight)

        self.softmax = torch.nn.Softmax(dim=1)
        self.softplus = torch.nn.Softplus(beta=self.temp, threshold=20)

        self.max_q_backup = max_q_backup
        self.deterministic_backup = deterministic_backup

        self.num_betas = max(int(num_total * (1. - mix_weight)), 1)
        self.num_rhos = int((num_total * mix_weight) // 3.)
        self.num_total = self.num_betas + self.num_rhos * 3
        self.mix_weight = (self.num_rhos * 3) / self.num_total

        print("self.num_betas: \t", self.num_betas)
        print('self.num_rhos: \t', self.num_rhos * 3)
        print('self.num_total: \t', self.num_total)
        print('self.mix_weight: %.2f\t'%self.mix_weight)

        # For implementation on the 
        self.discrete = False

    def _get_tensor_values(self, obs, actions, network=None):
        action_shape = actions.shape[0]
        obs_shape = obs.shape[0]
        num_repeat = int (action_shape / obs_shape)
        obs_temp = obs.unsqueeze(1).repeat(1, num_repeat, 1).view(obs.shape[0] * num_repeat, obs.shape[1])
        preds = network(obs_temp, actions)
        preds = preds.view(obs.shape[0], num_repeat, 1)
        return preds

    def _get_policy_actions(self, obs, num_actions, network=None):
        obs_temp = obs.unsqueeze(1).repeat(1, num_actions, 1).view(obs.shape[0] * num_actions, obs.shape[1])
        new_obs_actions, _, _, new_obs_log_pi, *_ = network(
            obs_temp, reparameterize=False, return_log_prob=True,
        )
        if not self.discrete:
            return new_obs_actions, new_obs_log_pi.view(obs.shape[0], num_actions, 1)
        else:
            return new_obs_actions

    def _calculate_log_prob(self, obs, num_actions, network=None):
        obs_temp = obs.unsqueeze(1).repeat(1, num_actions, 1).view(obs.shape[0] * num_actions, obs.shape[1])
        act_temp, _, _, new_obs_log_pi, *_ = self.policy(
            obs_temp, reparameterize=False, return_log_prob=True,
        )
        new_obs_log_pi = network.log_prob(obs_temp, act_temp)

        return new_obs_log_pi

    def train_from_torch(self, batch):
        self._current_epoch += 1

        rewards = batch['rewards']
        terminals = batch['terminals']
        obs = batch['observations']
        actions = batch['actions']
        next_obs = batch['next_observations']

        """
        Policy and Alpha Loss
        """
        new_obs_actions, policy_mean, policy_log_std, log_pi, *_ = self.policy(
            obs, reparameterize=True, return_log_prob=True,
        )

        # Entropy parameter tunning
        # alpha를 kl apaptive
        # alpha가 작으면 onestep + kl
        if self.use_automatic_entropy_tuning:
            alpha_loss = (self.log_alpha * ((-log_pi) - self.target_entropy).detach()).mean()
            self.alpha_optimizer.zero_grad()
            alpha_loss.backward()
            self.alpha_optimizer.step()
            alpha = self.log_alpha.exp()
        else:
            alpha_loss = 0
            alpha = 1

        if self.num_qs == 1:
            q_new_actions = self.qf1(obs, new_obs_actions)
        else:
            q_new_actions = torch.min(
                self.qf1(obs, new_obs_actions),
                self.qf2(obs, new_obs_actions),
            )

        """
        KL constraint to mix_weight * rho + (1 - mix_weight) * beta
        """
        # beta 가 rho가 되면 penalty가 0.5 -> alpha가 작아야하지 않을까
        # alpha가 작으면 kl이 beta에 가까워짐

        obs_temp = obs.unsqueeze(1).repeat(1, self.num_total, 1).view(obs.shape[0] * self.num_total, obs.shape[1])
        new_obs_actions_stack, _, _, log_pi_stack, *_ = self.policy(
            obs_temp, reparameterize=False, return_log_prob=True,
        )
        # B N D -> B*N D
        log_pi_stack = log_pi_stack.reshape((-1, self.num_total))
        log_pi = log_pi_stack.mean(dim=1)

        log_beta_stack = self.behavior_poliy.log_prob(obs_temp, new_obs_actions_stack).view(obs.shape[0], -1)
        log_random_stack = torch.ones_like(log_beta_stack) * np.log(0.5 ** new_obs_actions_stack.shape[-1])

        # Onestep 처럼 kl하고, alpha로 target_entropy만큼 유지
        kl_log_ratio = np.log(1 - self.mix_weight) + log_beta_stack - torch.logaddexp(np.log(1 - self.mix_weight) + log_beta_stack,
                                                                        np.log(self.mix_weight) + log_random_stack)
        kl_ratio = torch.exp(kl_log_ratio).detach()

        kl_const_loss = log_pi_stack - kl_ratio * log_beta_stack

        # policy_loss = - q_new_actions.mean() + alpha * kl_const_loss.mean()
        policy_loss = - q_new_actions.mean() + alpha * kl_const_loss.mean()

        if self._current_epoch < self.policy_eval_start:
            """
            For the initial few epochs, try doing behaivoral cloning, if needed
            conventionally, there's not much difference in performance with having 20k 
            gradient steps here, or not having it
            """
            policy_log_prob = self.policy.log_prob(obs, actions)
            policy_loss = (alpha * log_pi - policy_log_prob).mean()

        """
        QF Loss
        """

        q1_pred = self.qf1(obs, actions)
        if self.num_qs > 1:
            q2_pred = self.qf2(obs, actions)
        
        new_next_actions, _, _, new_log_pi, *_ = self.policy(
            next_obs, reparameterize=False, return_log_prob=True,
        )
        new_curr_actions, _, _, new_curr_log_pi, *_ = self.policy(
            obs, reparameterize=False, return_log_prob=True,
        )

        if not self.max_q_backup:
            if self.num_qs == 1:
                target_q_values = self.target_qf1(next_obs, new_next_actions)
            else:
                target_q_values = torch.min(
                    self.target_qf1(next_obs, new_next_actions),
                    self.target_qf2(next_obs, new_next_actions),
                )
            
            if not self.deterministic_backup:
                target_q_values = target_q_values - alpha * new_log_pi

        if self.max_q_backup:
            """when using max q backup"""
            next_actions_temp, _ = self._get_policy_actions(next_obs, num_actions=10, network=self.policy)
            target_qf1_values = self._get_tensor_values(next_obs, next_actions_temp, network=self.target_qf1).max(1)[0].view(-1, 1)
            target_qf2_values = self._get_tensor_values(next_obs, next_actions_temp, network=self.target_qf2).max(1)[0].view(-1, 1)
            target_q_values = torch.min(target_qf1_values, target_qf2_values)

        q_target = self.reward_scale * rewards + (1. - terminals) * self.discount * target_q_values
        q_target = q_target.detach()
            
        qf1_bellman = self.qf_criterion(q1_pred, q_target)
        if self.num_qs > 1:
            qf2_bellman = self.qf_criterion(q2_pred, q_target)

        ### add Ours

        # log E_{alpha * rho + (1-alpha) * beta} [exp Q]
        # 갯수 0이면 안하게
        beta_tensor, log_betas = self._get_policy_actions(obs, num_actions=self.num_betas, network=self.behavior_poliy)
        curr_actions_tensor, curr_log_pis = self._get_policy_actions(obs, num_actions=self.num_rhos, network=self.policy)
        new_curr_actions_tensor, new_log_pis = self._get_policy_actions(next_obs, num_actions=self.num_rhos, network=self.policy)
        random_actions_tensor = torch.FloatTensor(q1_pred.shape[0] * self.num_rhos, actions.shape[-1]).uniform_(-1, 1).to(ptu.device)

        q1_curr_actions = self._get_tensor_values(obs, curr_actions_tensor, network=self.qf1)
        q1_next_actions = self._get_tensor_values(obs, new_curr_actions_tensor, network=self.qf1)
        q1_rand = self._get_tensor_values(obs, random_actions_tensor, network=self.qf1)
        q1_beta = self._get_tensor_values(obs, beta_tensor, network=self.qf1)

        random_density = np.log(0.5 ** curr_actions_tensor.shape[-1])
        cat_q1 = torch.cat(
            [q1_rand,
             q1_next_actions + random_density - new_log_pis.detach(),
             q1_curr_actions + random_density - curr_log_pis.detach(),
             q1_beta], dim=1)

        s1_min_qf1_loss = torch.logsumexp(cat_q1 / self.temp, dim=1, ).mean()

        # - (1-alpha) * k * log E_beta [exp Q], k = E_beta [exp Q] / E_{alpha * rho + (1-alpha) * beta} [exp Q]
        # np.exp(x - logsumexp(x)), EX [g(X)] ≥ g(EX [X])
        with torch.no_grad():
            max_val = torch.logsumexp(cat_q1, dim=1, keepdim=True)
            # action에 대해서 mean
            k1 = torch.exp(q1_beta - max_val).mean(dim=1) / torch.exp(cat_q1 - max_val).mean(dim=1).detach()
        logsum = torch.logsumexp(q1_beta / self.temp, dim=1, )
        s2_min_qf1_loss = -((1-self.mix_weight) * k1 * logsum).mean()

        # - E_beta [alpha * rho / (alpha * rho + (1-alpha) * beta) Q]
        log_betas = log_betas.squeeze(-1)
        log_rho = torch.ones_like(log_betas) * random_density
        log_ratio = np.log(self.mix_weight) + log_rho - torch.logaddexp(np.log(1 - self.mix_weight) + log_betas,
                                                                        np.log(self.mix_weight) + log_rho)
        ratio = torch.exp(log_ratio).detach()
        s3_min_qf1_loss = -(ratio * q1_beta.squeeze(-1)).mean()

        # # - (E_pi[alpha * rho / (alpha * rho + (1-alpha) * beta) Q?] - E_beta[alpha * rho / (alpha * rho + (1-alpha) * beta)]) E_beta[Q]
        obs_temp = obs.unsqueeze(1).repeat(1, self.num_rhos, 1).view(obs.shape[0] * self.num_rhos, obs.shape[1])
        pi_log_beta = self.behavior_poliy.log_prob(obs_temp, curr_actions_tensor).view(obs.shape[0], -1)
        log_pi_rho = torch.ones_like(pi_log_beta) * random_density
        log_pi_ratio = np.log(self.mix_weight) + log_pi_rho - torch.logaddexp(np.log(1 - self.mix_weight) + pi_log_beta,
                                                                        np.log(self.mix_weight) + log_pi_rho)
        pi_ratio = torch.exp(log_pi_ratio).mean(dim=1).detach()
        beta_ratio = ratio.mean(dim=1).detach()
        s4_min_qf1_loss = -((pi_ratio - beta_ratio) * q1_beta.squeeze(-1).mean(dim=1)).mean()

        min_qf1_loss = s1_min_qf1_loss + s2_min_qf1_loss + s3_min_qf1_loss + s4_min_qf1_loss

        qf1_loss = qf1_bellman + min_qf1_loss * self.min_q_weight

        if self.num_qs > 1:
            # log E_{alpha * rho + (1-alpha) * beta} [exp Q]
            beta_tensor, log_betas = self._get_policy_actions(obs, num_actions=self.num_betas,
                                                              network=self.behavior_poliy)
            curr_actions_tensor, curr_log_pis = self._get_policy_actions(obs, num_actions=self.num_rhos,
                                                                         network=self.policy)
            new_curr_actions_tensor, new_log_pis = self._get_policy_actions(next_obs, num_actions=self.num_rhos,
                                                                            network=self.policy)
            random_actions_tensor = torch.FloatTensor(q2_pred.shape[0] * self.num_rhos, actions.shape[-1]).uniform_(-1,
                                                                                                                    1).to(
                ptu.device)

            q2_curr_actions = self._get_tensor_values(obs, curr_actions_tensor, network=self.qf2)
            q2_next_actions = self._get_tensor_values(obs, new_curr_actions_tensor, network=self.qf2)
            q2_rand = self._get_tensor_values(obs, random_actions_tensor, network=self.qf2)
            q2_beta = self._get_tensor_values(obs, beta_tensor, network=self.qf2)

            random_density = np.log(0.5 ** curr_actions_tensor.shape[-1])
            cat_q2 = torch.cat(
                [q2_rand,
                 q2_next_actions + random_density - new_log_pis.detach(),
                 q2_curr_actions + random_density - curr_log_pis.detach(),
                 q2_beta], dim=1)

            s1_min_qf2_loss = torch.logsumexp(cat_q2 / self.temp, dim=1, ).mean()

            # - (1-alpha) * k * log E_beta [exp Q], k = E_beta [exp Q] / E_{alpha * rho + (1-alpha) * beta} [exp Q]
            # np.exp(x - logsumexp(x)), EX [g(X)] ≥ g(EX [X])
            with torch.no_grad():
                max_val = torch.logsumexp(cat_q2, dim=1, keepdim=True)
                # actino에 대해서 mean
                k2 = torch.exp(q2_beta - max_val).mean(dim=1) / torch.exp(cat_q2 - max_val).mean(dim=1).detach()
            s2_min_qf2_loss = -((1 - self.mix_weight) * k2 * torch.logsumexp(q2_beta / self.temp, dim=1, )).mean()

            # - E_beta [alpha * rho / (alpha * rho + (1-alpha) * beta) Q]
            log_betas = log_betas.squeeze(-1)
            log_rho = torch.ones_like(log_betas) * random_density
            log_ratio = np.log(self.mix_weight) + log_rho - torch.logaddexp(np.log(1 - self.mix_weight) + log_betas,
                                                                            np.log(self.mix_weight) + log_rho)
            ratio = torch.exp(log_ratio).detach()
            s3_min_qf2_loss = -(ratio * q2_beta.squeeze(-1)).mean()

            # # - (E_pi[alpha * rho / (alpha * rho + (1-alpha) * beta)] - E_beta[alpha * rho / (alpha * rho + (1-alpha) * beta)]) E_beta[Q]
            obs_temp = obs.unsqueeze(1).repeat(1, self.num_rhos, 1).view(obs.shape[0] * self.num_rhos, obs.shape[1])
            pi_log_beta = self.behavior_poliy.log_prob(obs_temp, curr_actions_tensor).view(obs.shape[0], -1)
            log_pi_rho = torch.ones_like(pi_log_beta) * random_density
            log_pi_ratio = np.log(self.mix_weight) + log_pi_rho - torch.logaddexp(
                np.log(1 - self.mix_weight) + pi_log_beta,
                np.log(self.mix_weight) + log_pi_rho)
            pi_ratio = torch.exp(log_pi_ratio).mean(dim=1).detach()
            beta_ratio = ratio.mean(dim=1).detach()
            s4_min_qf2_loss = -((pi_ratio - beta_ratio) * q2_beta.squeeze(-1).mean(dim=1)).mean()

            min_qf2_loss = s1_min_qf2_loss + s2_min_qf2_loss + s3_min_qf2_loss + s4_min_qf2_loss

            qf2_loss = qf2_bellman + min_qf2_loss * self.min_q_weight


        """
        Update networks
        """
        self._num_policy_update_steps += 1
        self.policy_optimizer.zero_grad()
        policy_loss.backward(retain_graph=False)
        self.policy_optimizer.step()

        # Update the Q-functions iff 
        self._num_q_update_steps += 1
        self.qf1_optimizer.zero_grad()
        qf1_loss.backward(retain_graph=True)
        self.qf1_optimizer.step()

        if self.num_qs > 1:
            self.qf2_optimizer.zero_grad()
            qf2_loss.backward(retain_graph=True)
            self.qf2_optimizer.step()

        """
        Soft Updates
        """
        ptu.soft_update_from_to(
            self.qf1, self.target_qf1, self.soft_target_tau
        )
        if self.num_qs > 1:
            ptu.soft_update_from_to(
                self.qf2, self.target_qf2, self.soft_target_tau
            )
        """
        Save some statistics for eval
        """
        if self._need_to_update_eval_statistics:
            self._need_to_update_eval_statistics = False
            """
            Eval should set this to None.
            This way, these statistics are only computed for one batch.
            """

            self.eval_statistics['QF1 Bellman Loss'] = np.mean(ptu.get_numpy(qf1_bellman))
            self.eval_statistics['1st min QF1 Loss'] = np.mean(ptu.get_numpy(s1_min_qf1_loss))
            self.eval_statistics['k1'] = np.mean(ptu.get_numpy(k1))
            # self.eval_statistics.update(create_stats_ordered_dict(
            #     'q1_beta',
            #     ptu.get_numpy(q1_beta),
            # ))
            # self.eval_statistics.update(create_stats_ordered_dict(
            #     'logsum',
            #     ptu.get_numpy(logsum),
            # ))
            self.eval_statistics['2nd min QF1 Loss'] = np.mean(ptu.get_numpy(s2_min_qf1_loss))
            self.eval_statistics.update(create_stats_ordered_dict(
                'ratio',
                ptu.get_numpy(ratio),
            ))
            self.eval_statistics['3rd min QF1 Loss'] = np.mean(ptu.get_numpy(s3_min_qf1_loss))
            self.eval_statistics.update(create_stats_ordered_dict(
                'pi_ratio',
                ptu.get_numpy(pi_ratio),
            ))
            self.eval_statistics.update(create_stats_ordered_dict(
                'beta_ratio',
                ptu.get_numpy(beta_ratio),
            ))
            self.eval_statistics['pi_ratio - beta_ratio'] = np.mean(ptu.get_numpy(pi_ratio - beta_ratio))
            self.eval_statistics['ratio + pi_ratio - beta_ratio'] = np.mean(ptu.get_numpy(ratio + pi_ratio.unsqueeze(-1) - beta_ratio.unsqueeze(-1)))
            self.eval_statistics['4th min QF1 Loss'] = np.mean(ptu.get_numpy(s4_min_qf1_loss))
            self.eval_statistics['Total min QF1 Loss'] = np.mean(ptu.get_numpy(min_qf1_loss))
            self.eval_statistics['Total QF1 Loss'] = np.mean(ptu.get_numpy(qf1_loss))
            if self.num_qs > 1:
                self.eval_statistics['Bellman QF2 Loss'] = np.mean(ptu.get_numpy(qf2_bellman))
                self.eval_statistics['Min QF2 Loss'] = np.mean(ptu.get_numpy(min_qf2_loss))
                self.eval_statistics['Total QF2 Loss'] = np.mean(ptu.get_numpy(qf2_loss))

            if not self.discrete:
                self.eval_statistics.update(create_stats_ordered_dict(
                    'QF1 in-distribution values',
                    ptu.get_numpy(q1_curr_actions),
                ))
                self.eval_statistics.update(create_stats_ordered_dict(
                    'QF1 random values',
                    ptu.get_numpy(q1_rand),
                ))
                self.eval_statistics.update(create_stats_ordered_dict(
                    'QF1 next_actions values',
                    ptu.get_numpy(q1_next_actions),
                ))
                self.eval_statistics.update(create_stats_ordered_dict(
                    'actions', 
                    ptu.get_numpy(actions)
                ))
                self.eval_statistics.update(create_stats_ordered_dict(
                    'rewards',
                    ptu.get_numpy(rewards)
                ))

            self.eval_statistics['Num Q Updates'] = self._num_q_update_steps
            self.eval_statistics['Num Policy Updates'] = self._num_policy_update_steps
            self.eval_statistics.update(create_stats_ordered_dict(
                'Log beta',
                ptu.get_numpy(log_beta_stack + np.log(1 - self.mix_weight)),
            ))
            self.eval_statistics.update(create_stats_ordered_dict(
                'Log random',
                ptu.get_numpy(log_random_stack + np.log(self.mix_weight)),
            ))
            self.eval_statistics['KL ratio'] = np.mean(ptu.get_numpy(
                kl_ratio
            ))
            self.eval_statistics['KL Const. Loss'] = np.mean(ptu.get_numpy(
                kl_const_loss
            ))
            self.eval_statistics['Policy Loss'] = np.mean(ptu.get_numpy(
                policy_loss
            ))
            self.eval_statistics['Entropy'] = np.mean(ptu.get_numpy(
                -log_pi
            ))
            self.eval_statistics.update(create_stats_ordered_dict(
                'Q1 Predictions',
                ptu.get_numpy(q1_pred),
            ))
            if self.num_qs > 1:
                self.eval_statistics.update(create_stats_ordered_dict(
                    'Q2 Predictions',
                    ptu.get_numpy(q2_pred),
                ))
            self.eval_statistics.update(create_stats_ordered_dict(
                'Q Targets',
                ptu.get_numpy(q_target),
            ))

            self.eval_statistics.update(create_stats_ordered_dict(
                'Log Pis',
                ptu.get_numpy(log_pi),
            ))
            if not self.discrete:
                self.eval_statistics.update(create_stats_ordered_dict(
                    'Policy mu',
                    ptu.get_numpy(policy_mean),
                ))
                self.eval_statistics.update(create_stats_ordered_dict(
                    'Policy log std',
                    ptu.get_numpy(policy_log_std),
                ))
            
            if self.use_automatic_entropy_tuning:
                self.eval_statistics['Alpha'] = alpha.item()
                self.eval_statistics['Alpha Loss'] = alpha_loss.item()

            def get_all_state_overestim(paths):
                overestim = []
                path_over_info = {'path_overestim':[], 'gamm_return':[], 'q_val':[]}

                for path in paths:
                    gamma_return = 0
                    for i in reversed(range(path["rewards"].size)):
                        gamma_return = path["rewards"][i] + 0.99 * gamma_return * (1 - path["terminals"][i])
                        with torch.no_grad():
                            q_val = torch.stack([self.qf1(ptu.from_numpy(path["observations"][i]).unsqueeze(0),
                                               ptu.from_numpy(path["actions"][i]).unsqueeze(0)),
                                                 self.qf2(ptu.from_numpy(path["observations"][i]).unsqueeze(0),
                                                          ptu.from_numpy(path["actions"][i]).unsqueeze(0))], 0)
                            q_val = torch.min(q_val, dim=0)[0].squeeze(0).cpu().numpy()

                        path_over_info["gamm_return"].append(gamma_return.item())
                        path_over_info['q_val'].append((q_val).item())
                        path_over_info["path_overestim"].append((q_val - gamma_return).item())

                    path_over_info["gamm_return"] = path_over_info["gamm_return"][::-1]
                    path_over_info["q_val"] = path_over_info["q_val"][::-1]
                    path_over_info["path_overestim"] = path_over_info["path_overestim"][::-1]

                    overestim.append(path_over_info)

                return np.array(overestim)

            self._reserve_path_collector = MdpPathCollector(
                env=self.env, policy=self.policy,
            )
            self._reserve_path_collector.update_policy(self.policy)

            # Sampling
            eval_paths = self._reserve_path_collector.collect_new_paths(
                max_path_length=1000,
                num_steps=1000,
                discard_incomplete_paths=True,
            )

            overestim_info = get_all_state_overestim(eval_paths)

            self.eval_statistics.update(create_stats_ordered_dict('Overestimation', overestim_info[0]['path_overestim']))

        self._n_train_steps_total += 1

    def get_diagnostics(self):
        return self.eval_statistics

    def end_epoch(self, epoch):
        self._need_to_update_eval_statistics = True

    @property
    def networks(self):
        base_list = [
            self.policy,
            self.behavior_poliy,
            self.qf1,
            self.qf2,
            self.target_qf1,
            self.target_qf2,
        ]
        return base_list

    def get_snapshot(self):
        return dict(
            policy=self.policy,
            behavior_poliy=self.behavior_poliy,
            qf1=self.qf1,
            qf2=self.qf2,
            target_qf1=self.target_qf1,
            target_qf2=self.target_qf2,

        )

    def set_snapshot(self, snapshot):
        self.policy = snapshot['policy']
        self.behavior_poliy = snapshot['behavior_poliy']
        self.qf1 = snapshot['qf1']
        self.qf2 = snapshot['qf2']
        self.target_qf1 = snapshot['target_qf1']
        self.target_qf2 = snapshot['target_qf2']

