#%%
import os
import sys
import time
import json
from google import genai
import logging
from pathlib import PurePath
from google.genai.types import HttpOptions, Part


sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))
from utils.global_local_functions import generate_content_with_retry, load_questions


# Set environment variables for Google Cloud and Vertex AI
os.environ['GOOGLE_CLOUD_PROJECT'] = 'gen-lang-client-0982482224'  # Replace with your actual project ID
os.environ['GOOGLE_CLOUD_LOCATION'] = 'global'
os.environ['GOOGLE_GENAI_USE_VERTEXAI'] = 'True'
os.environ['GOOGLE_RESUMABLE_MEDIA_CHUNK_SIZE'] = str(10 * 1024 * 1024)



# FUNCTIONS ------------------------------------------------

def setup_logger(questions_file):
    # Create logs directory if it doesn't exist
    logs_dir = os.path.join("Global_Changes", "logs")
    os.makedirs(logs_dir, exist_ok=True)
    
    # Get current timestamp
    timestamp = time.strftime("%Y%m%d_%H%M%S")
    
    # Create log filename based on cubicle and timestamp
    base_name = PurePath(questions_file).stem
    log_filename = os.path.join(logs_dir, f"{timestamp}_{base_name}_results.log")
    
    # Configure logging
    logging.basicConfig(
        filename=log_filename,
        level=logging.INFO,
        format='%(asctime)s - %(levelname)s - %(message)s',
        filemode='w'  # Overwrite existing log file
    )
    
    # Add console handler to see logs in terminal too
    console = logging.StreamHandler()
    console.setLevel(logging.INFO)
    formatter = logging.Formatter('%(asctime)s - %(levelname)s - %(message)s')
    console.setFormatter(formatter)
    logging.getLogger('').addHandler(console)
    
    logging.info(f"Starting video understanding benchmark for {questions_file}")
    return log_filename


def generate_questions(Q_dict,video_number):
    logging.info(f"Generating questions for video number {video_number}")
    prompt_start = """

You are taking a multiple-choice benchmark.  
  
To answer the questions, you will be provided with two videos. The first video is the initial state of the scene, and the second video is the final state of the scene.
Please pay attention to the changes of objects in the scene between the two videos.
Furthermore, if we say an object had been removed, vanished, or disappeared from a cubicle it means it will not be present in the second video.
If we say an object has appeared in a cubicle it means it will be present in the second video.
If we say an object has been moved, it means it has moved within the cubicle or between cubicles.

Please answer the question to best of your ability. 

For each question below, reply with the single letter (A–E) that you believe is correct.  
Do not provide explanations—only the letter.

Please return a json object in the following format:

"answers": [
{
    "question": "Q1",
    "answer": "B"
},
{
    "question": "Q2",
    "answer": "C"
},
{
    "question": "Q3",
    "answer": "C"
},
{
    "question": "Q4",
    "answer": "B"
},
{
    "question": "Q5",
    "answer": "C"
},
{
    "question": "Q6",
    "answer": "C"
},
{
    "question": "Q7",
    "answer": "C"
}
....
]

Please use the Q# correspoding to the questions provided.
Also please always answer the questions, never return an empty json.
"""

    # Find all the questions that have the video number as the initial video
    questions_list = [key for key, value in Q_dict.items() if value["Initial Video"] == video_number]
    logging.info(f"Found {len(questions_list)} questions for video {video_number}")

    questions_prompt_list = []
    for i, question in enumerate(questions_list):
        questions_prompt_list.append(question + ":\n" + "Question: " + Q_dict[question]["Question"] + "\n")
        for choice in list(Q_dict[question]["Multiple Choice"].keys()):
            questions_prompt_list[i] += choice + ": " + Q_dict[question]["Multiple Choice"][choice] + "\n"

    prompt_end = """
"""

    questions_prompt = prompt_start + "\n".join(questions_prompt_list) + "\n" + prompt_end
    logging.debug(f"Generated prompt: {questions_prompt}")
    return questions_prompt



def save_results(questions_file, prompt, video_paths, questions_dict, dict_answers, total_correct, total_questions, model_name, temperature):
    logging.info(f"Saving results for {questions_file}")
    # Create output filename based on input filename with timestamp
    timestamp = time.strftime("%Y%m%d_%H%M%S")
    output_filename = os.path.splitext(questions_file)[0] + f"_{timestamp}_results.json"
    
    # Prepare the results dictionary
    results = {
        "model": model_name,
        "temperature": temperature,
        "prompt": prompt,
        "video_paths": video_paths,
        "questions": questions_dict,
        "answers": dict_answers,
        "score": {
            "correct": total_correct,
            "total": total_questions,
            "percentage": (total_correct / total_questions) * 100 if total_questions > 0 else 0
        },
        "timestamp": time.strftime("%Y-%m-%d %H:%M:%S")
    }
    
    # Save to JSON file
    with open(output_filename, 'w') as f:
        json.dump(results, f, indent=4)
    
    logging.info(f"Results saved to {output_filename}")


# INPUTS ------------------------------------------------

questions_file = "Global_Changes/Object_Counting_Questions.json"


gemini_model = "gemini-2.5-pro-preview-05-06"
temperature = 0.0
# Video path configuration
VIDEO_PATH_PATTERN = "gs://gemini-api-global-videos/episode_{i}_720p_10fps.mp4"


# CODE ------------------------------------------------

log_filename = setup_logger(questions_file)
logging.info("Logger initialized")
logging.info(f"Using Gemini model: {gemini_model}")
logging.info(f"Using questions file: {questions_file}")
# Initialize client without API key since we're using Vertex AI
client = genai.Client(http_options=HttpOptions(api_version="v1"))
logging.info("GenAI client initialized with Vertex AI")
questions_dict = load_questions(questions_file)
logging.info(f"Loaded {len(questions_dict)} questions from {questions_file}")

initial_video = 0
final_video = questions_dict[list(questions_dict.keys())[-1]]["Final Video"]
logging.info(f"Processing videos from {initial_video} to {final_video}")

dict_answers = {}
for i in range(initial_video, final_video):
    logging.info(f"Processing video pair {i} and {i+1}")
    
    video_1 = Part.from_uri(
        file_uri=VIDEO_PATH_PATTERN.format(i=i),
        mime_type="video/mp4")
    logging.info(f"Loaded initial video: episode_{i}_720p_10fps.mp4")
    
    video_2 = Part.from_uri(
        file_uri=VIDEO_PATH_PATTERN.format(i=i+1),
        mime_type="video/mp4")
    logging.info(f"Loaded final video: episode_{i+1}_720p_10fps.mp4")

    prompt = generate_questions(questions_dict,i)
    logging.info("Generated questions prompt")
    logging.info(f"Prompt being sent to Gemini:\n{prompt}")
    
    result = generate_content_with_retry(client, gemini_model, [video_1, video_2, prompt], temperature=temperature)
    logging.info("Received response from Gemini model")
    logging.info(f"Raw response from Gemini: {result.text}")
    
    # Check if response is empty
    if not result or not result.text:
        logging.error("Empty response received from Gemini model")
        raise ValueError("Empty response from Gemini model")
        
    try:
        answers_json = json.loads(result.text)
        logging.info(f"Successfully parsed JSON response: {answers_json}")
    except json.JSONDecodeError as e:
        logging.error(f"Failed to parse JSON response: {e}")
        logging.error(f"Raw response that failed to parse: {result.text}")
        raise

    for answer in answers_json['answers']:
        dict_answers[answer['question']] = answer['answer']
    
    # for f in client.files.list():
    #     client.files.delete(name=f.name)
    #     logging.info(f"Deleted temporary file: {f.name}")

logging.info("Completed processing all video pairs")
logging.info(f"Final answers dictionary: {dict_answers}")

total_correct = 0
total_questions = 0

for question in questions_dict:
    total_questions += 1
    if questions_dict[question]["Correct Choice"] == dict_answers[question]:
        total_correct += 1

logging.info(f"Scoring complete: {total_correct} correct out of {total_questions} questions")

# Collect video paths
video_paths = []
for i in range(initial_video, final_video):
    video_paths.append(VIDEO_PATH_PATTERN.format(i=i))
    video_paths.append(VIDEO_PATH_PATTERN.format(i=i+1))
logging.info(f"Collected {len(video_paths)} video paths")

# Save all results
save_results(
    questions_file=questions_file,
    prompt=prompt,
    video_paths=video_paths,
    questions_dict=questions_dict,
    dict_answers=dict_answers,
    total_correct=total_correct,
    total_questions=total_questions,
    model_name=gemini_model,
    temperature=temperature
)
logging.info("Benchmark execution completed successfully")


