import jax
import jax.numpy as jnp
import optax
from approxml.mnist import load_model, get_loader
from approxml.simulators import gan_simulator, mvt_norm_simulator
from approxml.utils import gen_simulation_samples, grad_log_normal
from approxml.scorematching import fit_linear_sm
from functools import partial
import pickle

LATENT_DIM = 50
N_RUNS = 100
N_ITER = 500
N_PROP = 150
N_SIM_DST = 150
N_PARAM_DIM = LATENT_DIM
N_DATA_DIM = 256
PRIOR_SIGMA = 1e-1

gen, gen_state = load_model("./saved_models", 
                            "generator_1.eqx", 
                            "generator_state_1.eqx", 
                            jax.random.PRNGKey(42), 
                            LATENT_DIM, 
                            (16,16,1))

gan_simulator = partial(gan_simulator, 
                        generator=gen, 
                        state=gen_state,
                        latent_dim=LATENT_DIM,
                        sigma=PRIOR_SIGMA)

mnist_loader = get_loader((16,16,1),
                          36)
images_batch , _ = next(iter(mnist_loader))

def run_sgd(key, 
            theta_init, 
            obs,
            sigma,
            simulator_fn,
            learning_rate,
            lamb_val):
    obs_aug = jnp.concatenate([obs, jnp.ones_like(obs[..., :1])], axis=-1) 
    
    optimizer = optax.chain(
        optax.clip_by_global_norm(1.0),
        optax.adam(learning_rate=learning_rate)
    )
    opt_state = optimizer.init(theta_init)

    @jax.jit
    def update(params, opt_state, key, step):
        key, subkey = jax.random.split(key)
        current_sigma = sigma
        current_prop_cov = current_sigma * jnp.eye(N_PARAM_DIM)
        current_gen_sim_fn = partial(gen_simulation_samples,
            simulator_fn=simulator_fn,
            prop_sim_fn=partial(mvt_norm_simulator, cov=current_prop_cov),
            n_prop=N_PROP,
            n_sim_dst=N_SIM_DST)
        
        current_grad_fn = partial(fit_linear_sm,
            gen_sim_fn=current_gen_sim_fn,
            grad_log_prop_fn=partial(grad_log_normal, cov=current_prop_cov),
            n_sim_dst=N_SIM_DST,
            n_prop=N_PROP,
            lamb=lamb_val)
        
        W, _, _, _ = current_grad_fn(subkey, params)
        grads = jnp.einsum('mk,ik->im', W.T, obs_aug).sum(0)
        grads = - grads  
        updates, opt_state = optimizer.update(grads, opt_state, params)
        params = optax.apply_updates(params, updates)
        return params, opt_state, key, grads, current_sigma

    theta_values = []
    grad_values = []
    sigma_values = []

    theta = theta_init.copy()
    for i in range(N_ITER):
        theta, opt_state, key, grads, current_sigma = update(theta, opt_state, key, i)
        theta_values.append(theta.copy()) 
        grad_values.append(grads.copy())  
        sigma_values.append(current_sigma) 

    return {
        'theta_values': jnp.array(theta_values),
        'grad_values': jnp.array(grad_values),
        'sigma_values': jnp.array(sigma_values),
    }

for lamb_val in [1e-1, 5e-1, 1e0, 5e0, 1e1]:
    obs_idx = 0
    print("Running lambda: ", lamb_val)
    obs = images_batch[obs_idx].reshape(1, 256)
    obs = jnp.array(obs)

    theta_init = jnp.zeros(N_PARAM_DIM)
    prop_sigma = 0.2
    lr = 5e-2

    res_list = []
    err_list = []

    key = jax.random.PRNGKey(0)

    for i in range(N_RUNS):
        key, subkey = jax.random.split(key)
        res = run_sgd(subkey, 
                    theta_init, 
                    obs, 
                    prop_sigma, 
                    gan_simulator, 
                    lr,
                    lamb_val)
        
        sims = gan_simulator(jax.random.PRNGKey(42), 
                            res['theta_values'][-300:,:].mean(0),
                            100)
        
        err = jnp.linalg.norm(sims - obs, axis=1).mean()
        print(f"SM Run {i} error: {err}")

        err_list.append(err)
        res_list.append(res)

    with open(f'3_mnist_sm_lamb_{lamb_val}.pkl', 'wb') as f:
        pickle.dump({
            'res_list': res_list,
            'err_list': err_list
        }, f)