# coding=utf-8
# Copyright 2022 The Conceptual Learning Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Library for reading DatasetSpec and DataSuiteSpec files."""

from conceptual_learning.cscan import inputs
from conceptual_learning.cscan.specs import dataset_specs
from conceptual_learning.cscan.specs import dataset_suite_specs


def is_valid_dataset_spec_id(dataset_spec_id):
  """Returns whether the given spec ID has a corresponding DatasetSpec."""
  return dataset_spec_id in dataset_specs.DATASET_SPEC_BY_ID


def load_dataset_spec(dataset_spec_id):
  """Returns a spec with the given id from `dataset_specs.json`.

  Args:
    dataset_spec_id: Id of the DatasetSpec to load.

  Raises:
    ValueError: If no DatasetSpec with the given id can be found.
  """
  if not is_valid_dataset_spec_id(dataset_spec_id):
    raise ValueError(f'Invalid dataset spec id: {dataset_spec_id}')
  return dataset_specs.DATASET_SPEC_BY_ID[dataset_spec_id]


def load_dataset_suite_spec(
    dataset_suite_spec_id):
  """Returns a spec with the given id from `dataset_suite_specs.json`.

  Args:
    dataset_suite_spec_id: Id of the DatasetSuiteSpec to load.

  Raises:
    ValueError: If no DatasetSuiteSpec with the given id can be found.
  """
  dataset_suite_spec_by_id = dataset_suite_specs.DATASET_SUITE_SPEC_BY_ID
  if dataset_suite_spec_id not in dataset_suite_spec_by_id:
    raise ValueError(f'Invalid dataset suite spec id: {dataset_suite_spec_id}')
  return dataset_suite_spec_by_id[dataset_suite_spec_id]
