# coding=utf-8
# Copyright 2022 The Conceptual Learning Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Library for checking consistency between productions.

Unreliable rules could cause contradictory examples to be generated.  By this
we mean more than directly contradictory examples such as "[walk] = WALK" and
"[walk] = RUN", but also rules that can be derived from other rules.

For example, if we accept "[walk] = WALK" and "[x twice] = [x] [x]" to be
true, then we _must_ accept that the derived rule "[walk twice] = WALK WALK" is
also true, and therefore "[walk twice] = WALK WALK WALK" would be considered
a contradictory example.  If the first two rules above have already been added
to the context, then we should reject the addition of the last rule to the same
context.

This module implements the idea of an "inference engine", which in its current
implementation incrementally and aggressively calculates rules that must be true
based on all the rules added to the context so far.  The dataset_generation.py
module uses this to filter out examples that are contradictory in order to
guarantee context examples are consistent.

"""

import collections
import dataclasses
import logging
from typing import Dict, List, Optional, Set, Tuple

import nltk

from conceptual_learning.cscan import conceptual_learning as cl
from conceptual_learning.cscan import nltk_utils
from conceptual_learning.cscan import production_composition

_InputTokens = Tuple[str, Ellipsis]
_OutputTokens = Tuple[str, Ellipsis]


class InconsistencyError(Exception):
  """The proposed collection of productions leads to contradictory examples."""


@dataclasses.dataclass
class Inconsistency:
  """Inconsistency information.

  Attributes:
    type: A label of Qualifier.M or Qualifier.D indicating the type of
    inconsistency that has been detected:
      - Qualifier.M: There is an inconsistency with a known monotonic
        production.
      - Qualifier.D: There is an inconsistency with a known defeasible
        production, and there is no inconsistency with known monotonic
        productions.
    existing_inconsistency: The production that directly contradicts an incoming
      production or composition thereof.
    incoming_inconsistency: The production being added to the inference engine,
      or composition thereof, that causes a direct contradiction with an
      existing production.
    existing_inconsistency_source: The source productions of
      `existing_inconsistency`. If `existing_inconsistency` is a source
      production, then this set will contain a single element which is
      `existing_inconsistency`.
    incoming_inconsistency_source: The source productions of
      `incoming_inconsistency`. If `incoming_inconsistency` is the same
      production that was added (i.e. no composition steps were needed to get
      `incoming_inconsistency`) then this set will contain a single element
      which is `existing_inconsistency`.
  """
  type: cl.Qualifier
  existing_inconsistency: nltk.grammar.Production
  incoming_inconsistency: nltk.grammar.Production
  existing_inconsistency_source: Set[nltk.grammar.Production]
  incoming_inconsistency_source: Set[nltk.grammar.Production]


@dataclasses.dataclass
class Implication:
  """Implication information.

  Attributes:
    type: The type of implication. This attribute is determined based on the
      qualifier of the implication set:
        - cl.Qualifier.M: All the productions in source_productions are in
          InferenceEngine.monotonic_productions
        - cl.Qualifier.D: There exist at least one production in
          source_productions that is not in
          InferenceEngine.monotonic_productions
    production: The production that is implied by `source_productions`.
    source_productions: The source productions that imply `production`, i.e.
      performing inference on `source_productions` yields, among others,
      `production`.
  """
  type: cl.Qualifier
  production: nltk.grammar.Production
  source_productions: Set[nltk.grammar.Production]


@dataclasses.dataclass
class AddProductionStatus:
  """Status of trying to add a production to the inference engine.

  Attributes:
    inconsistency: Inconsistency observed when adding the production. `None`
      indicates that there was no inconsistency when adding the production.
    implication: Implication if the production to be added was already in the
      inference engine. `None` indicates that there was no implication.
  """
  inconsistency: Optional[Inconsistency] = None
  implication: Optional[Implication] = None


def _extract_input_tokens_and_output_tokens(
    production):
  input_tokens = tuple(nltk_utils.extract_rhs_tokens(production))
  output_tokens = tuple(nltk_utils.extract_lhs_tokens(production))
  return input_tokens, output_tokens


@dataclasses.dataclass
class InferenceEngine:
  """The class for checking example set consistency.

  The current implementation checks against contradictory examples by
  maintaining a mapping from input tokens to output tokens.  An example is
  detected as contradictory when it maps the same input tokens to different
  output tokens than the one already recorded.

  By calling the public method add_production, the inference engine's states are
  updated with increasingly large collections of added and inferred productions
  that are consistent.

  Specifically, an inference engine maintains collections of productions:
  monotonic_productions and all_productions, satisfying the following
  properties:

  - monotonic_productions is a subset of all_productions.
  - monotonic_productions is closed under composition: if p and q are composable
    productions in monotonic_productions, then so is their composition at every
    composable index.
  - all_productions is closed under composition.

  We then deem productions in the difference between all_productions and
  monotonic_productions as the collection of "defeasible productions", which
  contains exactly those productions that cannot be expressed as the composition
  of monotonic productions.

  The inference engine instance also tracks a collection of "source productions"
  consisting of productions that are added by a call to the add_production
  method.  This is different from the production provenance tracking done by the
  provenance_by_production field, which tracks all the productions during the
  course of context generation.  For example, hidden rules are typically not
  included in the inference engine's source productions, but are considered
  source productions by production provenance tracking.

  Attributes:
    source_productions: Productions that are added to the inference engine by a
      call to the add_production method without causing inconsistencies.
    monotonic_productions: Productions that are monotonic.
    all_productions: All productions that are known to be true to the inference
      engine.
    _productions_by_input_tokens: Mapping from the input token sequences to a
      set of productions.
    _productions_by_lhs_symbol: Mapping from nonterminal symbols to the set of
      productions with the symbol on the LHS.  This is to speed up lookups.
    _productions_by_index_by_rhs_symbol: Mapping from nonterminal symbols to the
      mapping from the RHS index at which the symbol appears to the set of
      productions.  This is to speed up lookups.
    _source_productions_by_production: Mapping from productions to the list of
      source productions from which the production is built by calling
      production_composition.compose.  A production in the inference engine is
      considered a "source production" if it is added to the inference engine
      with the add_production method.  Productions in the list are not deduped.
    _productions_by_num_variables: Mapping from numbers of variables to the list
      of productions with that that number of variables.
    track_multiple_provenances: Whether to track a single provenance for each
      production or more than one. If True, then provenances will be populated
      in provenances_by_production only; if False, then they will be populated
      in provenance_by_production only.
    provenance_by_production: A ProductionProvenanceMapping to record production
      provenances of all compositions taking place during calls to the
      add_production method.
    provenances_by_production: A mapping from each production to a set of
      production provenances which composing them leads to that production.
    _inconsistency_by_known_production: Mapping from productions that are not
      added to the inference engine to its inconsistency or None according to
      the current states of the inference engine.  This cache is used by the
      inconsistency_from_adding_production method and is cleared every time a
      new production is successfully added.  It is not backed up by the
      backup_states method.
    defeasible_productions: Read-only view on all the defeasible productions.
  """
  # The ordering in which the productions are added to the collections is
  # not recorded, since we have not needed this so far.
  source_productions: Set[nltk.grammar.Production] = dataclasses.field(
      default_factory=set)
  monotonic_productions: Set[nltk.grammar.Production] = dataclasses.field(
      default_factory=set)
  all_productions: Set[nltk.grammar.Production] = dataclasses.field(
      default_factory=set)

  # In CSCAN every input token sequence has at most one production if there's
  # no inconsistency within the inference engine; otherwise, a sequence of input
  # tokens can have one or more productions.
  _productions_by_input_tokens: Dict[_InputTokens,
                                     Set[nltk.grammar.Production]] = (
                                         dataclasses.field(
                                             default_factory=dict))
  _productions_by_lhs_symbol: Dict[
      str, Set[nltk.grammar.Production]] = dataclasses.field(
          default_factory=lambda: collections.defaultdict(set))
  _productions_by_index_by_rhs_symbol: Dict[str, Dict[
      int, Set[nltk.grammar.Production]]] = dataclasses.field(
          default_factory=lambda: collections.defaultdict(dict))
  _source_productions_by_production: Dict[
      nltk.grammar.Production,
      List[nltk.grammar.Production]] = dataclasses.field(default_factory=dict)
  _productions_by_num_variables: Dict[int, Set[nltk.grammar.Production]] = (
      dataclasses.field(default_factory=dict))

  track_multiple_provenances: bool = False


  # Production provenance records, its state is not restored by the backup and
  # restore methods, but the reference to the records is maintained during
  # backup and restore.
  provenance_by_production: (
      production_composition.ProductionProvenanceMapping) = (
          dataclasses.field(
              default_factory=production_composition.ProductionProvenanceDict,
              compare=False))

  provenances_by_production: (
      production_composition.ProductionProvenancesDict) = (
          dataclasses.field(
              default_factory=production_composition.ProductionProvenancesDict,
              compare=False))

  # Cache, not backed up/restored by the backup/restore methods.
  _inconsistency_by_known_production: Dict[nltk.grammar.Production,
                                           Inconsistency] = (
                                               dataclasses.field(
                                                   default_factory=dict,
                                                   init=False))

  @property
  def defeasible_productions(self):
    return self.all_productions - self.monotonic_productions

  def _production_is_defeasible(self,
                                production):
    # This is more efficient than calling the defeasible_productions property.
    return (production in self.all_productions and
            production not in self.monotonic_productions)

  def _update_states(self,
                     production,
                     is_monotonic = False):
    """Updates the inference engine's states.

    This should be called only by add_production or similar methods (e.g.,
    copy_monotonic_engine) after any relevant consistency checks have been
    performed.

    Args:
      production: The production to be added to the inference engine.
      is_monotonic: Whether the production should be added to the
        monotonic_productions collection (i.e., whether the production is being
        asserted as monotonically true).
    """
    self.all_productions.add(production)
    num_variables = production_composition.num_variables_from_production(
        production)
    self._productions_by_num_variables.setdefault(num_variables,
                                                  set()).add(production)

    if is_monotonic:
      self.monotonic_productions.add(production)

    input_tokens, _ = _extract_input_tokens_and_output_tokens(production)
    self._productions_by_input_tokens.setdefault(input_tokens,
                                                 set()).add(production)

    lhs_symbol = production.lhs()[nltk.grammar.TYPE]
    self._productions_by_lhs_symbol[lhs_symbol].add(production)

    for i, item in enumerate(production.rhs()):
      if isinstance(item, nltk.grammar.Nonterminal):
        rhs_symbol = item[nltk.grammar.TYPE]
        self._productions_by_index_by_rhs_symbol[rhs_symbol].setdefault(
            i, set()).add(production)

  def _composable_indices_and_productions_for_production_as_parent(
      self, production
  ):
    """Returns list of composable (index, other_parent) with production."""
    indices_and_productions = []

    # The production can be composed as parent when other_parent's LHS appears
    # in the production's RHS.
    for i, item in enumerate(production.rhs()):
      if isinstance(item, nltk.grammar.Nonterminal):
        rhs_symbol = item[nltk.grammar.TYPE]
        for other_parent in self._productions_by_lhs_symbol[rhs_symbol]:
          indices_and_productions.append((i, other_parent))

    return indices_and_productions

  def _composable_indices_and_productions_for_production_as_other_parent(
      self, production
  ):
    """Returns list of composable (index, parent) with production."""
    indices_and_productions = []

    # The production can be composed as other_parent when its LHS appears in
    # parent's RHS.
    lhs_symbol = production.lhs()[nltk.grammar.TYPE]
    for index, parents in (
        self._productions_by_index_by_rhs_symbol[lhs_symbol].items()):
      for parent in parents:
        indices_and_productions.append((index, parent))

    return indices_and_productions

  def copy_monotonic_engine(self):
    """Returns a copy of just the monotonic portion of the inference engine.

    New containers fields are created, but the contained productions themselves
    are only shallow copied.
    """
    monotonic_source_productions = set(
        filter(lambda p: p in self.monotonic_productions,
               self.source_productions))

    source_productions_by_production = {}
    for production, source_productions in (
        self._source_productions_by_production.items()):
      if production in self.monotonic_productions:
        source_productions_by_production[production] = source_productions[:]

    # Note that we don't need to copy the ProductionProvenance object itself
    # because it is a frozen dataclass and will never be modified once created.
    provenance_by_production = dict(
        filter(lambda k_v: k_v[0] in self.monotonic_productions,
               self.provenance_by_production.items()))

    provenances_by_production = (
        production_composition.ProductionProvenancesDict())
    for production, provenances in self.provenances_by_production.items():
      if production in self.monotonic_productions:
        provenances_by_production[production] = provenances

    copied = InferenceEngine(
        track_multiple_provenances=self.track_multiple_provenances,
        source_productions=monotonic_source_productions,
        _source_productions_by_production=source_productions_by_production,
        provenance_by_production=provenance_by_production,
        provenances_by_production=provenances_by_production)
    for production in self.monotonic_productions:
      copied._update_states(production, True)
    return copied

  def backup_states(self):
    """Returns a copy of the inference engine.

    New containers fields are created, but the contained productions themselves
    are only shallow copied.
    """
    # We handle the fields directly to avoid deepcopying the productions.
    productions_by_lhs_symbol_dict = {}
    for symbol, productions in self._productions_by_lhs_symbol.items():
      productions_by_lhs_symbol_dict[symbol] = set(productions)

    productions_by_index_by_rhs_symbol_dict = {}
    for symbol, productions_by_index in (
        self._productions_by_index_by_rhs_symbol.items()):
      productions_by_index_by_rhs_symbol_dict.setdefault(symbol, {})
      for index, productions in productions_by_index.items():
        productions_by_index_by_rhs_symbol_dict[symbol][index] = (
            set(productions))

    source_productions_by_production = {}
    for production, source_productions in (
        self._source_productions_by_production.items()):
      source_productions_by_production[production] = source_productions[:]

    productions_by_num_variables = {}
    for num_variables, productions in (
        self._productions_by_num_variables.items()):
      productions_by_num_variables[num_variables] = set(productions)

    productions_by_input_tokens = {}
    for input_tokens, productions in self._productions_by_input_tokens.items():
      productions_by_input_tokens[input_tokens] = set(productions)

    backup = InferenceEngine(
        track_multiple_provenances=self.track_multiple_provenances,
        source_productions=set(self.source_productions),
        monotonic_productions=set(self.monotonic_productions),
        all_productions=set(self.all_productions),
        _productions_by_input_tokens=productions_by_input_tokens,
        _productions_by_lhs_symbol=collections.defaultdict(
            set, productions_by_lhs_symbol_dict),
        _productions_by_index_by_rhs_symbol=collections.defaultdict(
            dict, productions_by_index_by_rhs_symbol_dict),
        _source_productions_by_production=source_productions_by_production,
        _productions_by_num_variables=productions_by_num_variables,
        provenance_by_production=self.provenance_by_production,
        provenances_by_production=self.provenances_by_production)
    return backup

  def _restore_states(self, backup):
    for field in dataclasses.fields(self):
      setattr(self, field.name, getattr(backup, field.name))

  def get_contradicting_productions(
      self,
      production):
    """Returns a set of productions contradicting the given production, if any.

    Args:
      production: The production to check.
    """
    input_tokens, output_tokens = _extract_input_tokens_and_output_tokens(
        production)
    known_productions = self._productions_by_input_tokens.get(
        input_tokens, set())
    contradicting_productions = set()
    for known_production in known_productions:
      _, known_output_tokens = _extract_input_tokens_and_output_tokens(
          known_production)
      if output_tokens != known_output_tokens:
        contradicting_productions.add(known_production)
    return contradicting_productions

  def get_source_productions(
      self, production
  ):
    """Returns the source productions of a production."""
    if production not in self.all_productions:
      return None

    return self._source_productions_by_production[production]

  def get_productions_of_num_variables(
      self, num_variables):
    return self._productions_by_num_variables.get(num_variables, set())

  def get_productions_of_lhs_symbol(
      self, symbol):
    return self._productions_by_lhs_symbol.get(symbol, set())

  def _compose_and_record_source_productions(
      self, parent,
      other_parent,
      index):
    """Returns the composed production and records the source productions."""
    if self.track_multiple_provenances:
      composed_production = production_composition.compose(
          parent,
          other_parent,
          index,
          provenances_by_production=self.provenances_by_production)
    else:
      composed_production = production_composition.compose(
          parent,
          other_parent,
          index,
          provenance_by_production=self.provenance_by_production)

    parent_source_productions = self.get_source_productions(parent)
    other_parent_source_productions = self.get_source_productions(other_parent)

    if (parent_source_productions is None or
        other_parent_source_productions is None):
      raise ValueError('Composing productions not in inference engine.')

    # composed_productions's source_productions will be updated only if it does
    # not have source productions yet, or we have found a shorter list of source
    # productions.  In particular, if composed_production happens to be a source
    # production itself, its source_productions list will remain the singleton
    # list consisting of itself.
    current_source_productions = (
        parent_source_productions + other_parent_source_productions)
    existing_source_productions = self.get_source_productions(
        composed_production)
    if (existing_source_productions is None or
        (len(current_source_productions) < len(existing_source_productions))):
      self._source_productions_by_production[composed_production] = (
          current_source_productions)

    return composed_production

  def _add_production(self,
                      production,
                      is_monotonic = False,
                      exhaustive = False):
    """Returns the status of adding the production.

    This method always updates the state of the inference engine, so we should
    keep this method private, and other methods calling it should handle
    restoring the states as needed.

    Args:
      production: The production to be added to the inference engine.
      is_monotonic: Whether the production should be added to the
        monotonic_productions collection (i.e., whether the production is being
        asserted as monotonically true).
      exhaustive: The mode that controls how we search the composable
        productions space. If True then the search will continue as long as
        there are composable productions. Otherwise, we stop the search if a
        monotonic inconsistency was detected or if a defeasible inconsistency
        was detected and there are no more monotonic productions to check.
    """
    status = AddProductionStatus()
    implication = None
    if (production in self.all_productions and
        production not in self.source_productions):
      implication_source = set(
          self._source_productions_by_production[production])
      # Find the implication type. Note that the type of the implication
      # represents the type of implication so far. Specifically, a known
      # defeasible production can become a monotonic one.
      if production in self.monotonic_productions:
        implication_type = cl.Qualifier.M
      else:
        implication_type = cl.Qualifier.D
      implication = Implication(
          type=implication_type,
          production=production,
          source_productions=implication_source)
      status.implication = implication

    # Whenever a production is manually added, it is its own source production.
    self.source_productions.add(production)
    self._source_productions_by_production[production] = [production]

    if self.track_multiple_provenances:
      self.provenances_by_production.add_provenance(
          production,
          production_composition.ProductionProvenance(source=production))
    # We always check all the descendants of the production just added since
    # their source_productions list might need to be updated.

    # We keep track of whether or not the production should be added to
    # monotonic_productions in order to always add monotonic productions first.
    productions_to_check_by_is_monotonic = {True: set(), False: set()}
    productions_to_check_by_is_monotonic[is_monotonic].add(production)

    # We want to first detect monotonic inconsistencies, that is, adding the
    # production as if there were no defeasible productions in the inference
    # engine.  In principle we could have implemented this with nested inference
    # engines, one containing only monotonic productions and the other
    # containing all productions.  With the current implementation this is done
    # by not returning immediately when the production causes only defeasible
    # inconsistency until all the possible monotonic productions have been
    # checked.
    while any(productions_to_check_by_is_monotonic.values()):
      if productions_to_check_by_is_monotonic[True]:
        current_production_is_monotonic = True
        current_production = productions_to_check_by_is_monotonic[True].pop()
      else:
        current_production_is_monotonic = False
        current_production = productions_to_check_by_is_monotonic[False].pop()

      maybe_contradicting_productions = self.get_contradicting_productions(
          current_production)
      for maybe_contradicting_production in maybe_contradicting_productions:
        existing_inconsistency_source = set(
            self
            ._source_productions_by_production[maybe_contradicting_production])
        incoming_inconsistency_source = set(
            self._source_productions_by_production[current_production])

        if (current_production_is_monotonic and
            maybe_contradicting_production in self.monotonic_productions):
          if (status.inconsistency is None or
              status.inconsistency.type == cl.Qualifier.D):
            # Only keep track of the first monotonic inconsistency.
            # If the previously encountered inconsistency was defeasible then we
            # ignore it and record the monotonic one.
            monotonic_inconsistency = Inconsistency(
                type=cl.Qualifier.M,
                existing_inconsistency=maybe_contradicting_production,
                incoming_inconsistency=current_production,
                existing_inconsistency_source=existing_inconsistency_source,
                incoming_inconsistency_source=incoming_inconsistency_source)
            status.inconsistency = monotonic_inconsistency
          if not exhaustive:
            # If the inconsistency happens entirely within monotonic productions
            # and exhaustive flag is False, we can immediately return with a
            # monotonic inconsistency.
            return status
        else:
          # Otherwise we just remember the fact that there is a defeasible
          # inconsistency detected, and return it only after all the monotonic
          # productions have been checked.
          if status.inconsistency is None:
            defeasible_inconsistency = Inconsistency(
                type=cl.Qualifier.D,
                existing_inconsistency=maybe_contradicting_production,
                incoming_inconsistency=current_production,
                existing_inconsistency_source=existing_inconsistency_source,
                incoming_inconsistency_source=incoming_inconsistency_source)
            status.inconsistency = defeasible_inconsistency

      # When current_production is monotonic, this _update_states call could
      # leave the inference engine in a somewhat inconsistent state, where all
      # the monotonic productions are consistent, but contain productions that
      # contradict some defeasible productions.
      self._update_states(current_production, current_production_is_monotonic)

      for index, other_parent in (
          self._composable_indices_and_productions_for_production_as_parent(
              current_production)):
        composed_production = self._compose_and_record_source_productions(
            current_production, other_parent, index)
        composed_production_is_monotonic = (
            current_production_is_monotonic and
            other_parent in self.monotonic_productions)
        # There are only three cases where the composed production needs to be
        # processed:
        # - If self.track_multiple_provenances is True. This is because the fact
        #   that this production was detected means that there might be a new
        #   provenace for this production (and subsequently all productions with
        #   provenances that have this production as an intermediate
        #   production).
        # - If it is a known defeasible production, but now we know it is in
        #   fact monotonic (as the composition of two monotonic parents), then
        #   we promote it to monotonic_productions.  This could result in other
        #   productions needing to be promoted, so we include it in the queue
        #   of productions to check next.
        # - If it is not in all_productions, then we need to check its
        #   consistency.
        # The other possible scenarios that do not require any action since the
        # inference engine's state would not change (when
        # self.track_multiple_provenances is False) are:
        # - If the composed production is already a known monotonic production.
        # - If the composed production is already a known defeasible production,
        #   and we still do not know if it can be the composition of two
        #   monotonic parents.
        if ((self._production_is_defeasible(composed_production) and
             composed_production_is_monotonic) or composed_production
            not in self.all_productions) or self.track_multiple_provenances:
          productions_to_check_by_is_monotonic[
              composed_production_is_monotonic].add(composed_production)

      for index, parent in (
          self
          ._composable_indices_and_productions_for_production_as_other_parent(
              current_production)):
        composed_production = self._compose_and_record_source_productions(
            parent, current_production, index)
        composed_production_is_monotonic = (
            current_production_is_monotonic and
            parent in self.monotonic_productions)
        # Below is the same logic as the case above.
        if ((self._production_is_defeasible(composed_production) and
             composed_production_is_monotonic) or composed_production
            not in self.all_productions) or self.track_multiple_provenances:
          productions_to_check_by_is_monotonic[
              composed_production_is_monotonic].add(composed_production)

      # If there are no monotonic productions left to check after including
      # current_production's children, defeasible inconsistency was already
      # detected, and we're not performing exhaustive check, we can just return
      # the defeasible inconsistency.
      if (not productions_to_check_by_is_monotonic[True] and
          status.inconsistency is not None and not exhaustive):
        return status
    return status

  def _clear_inconsistency_by_known_production(self):
    """Clears known inconsistencies cache."""
    # Productions which are known to cause monotonic inconsistencies will
    # remain so after adding a new production to the inference engine, while
    # productions that used to cause defeasible inconsistencies could now
    # cause monotonic inconsistenties.  To keep the code simpler we clear the
    # whole cache here until this is identified as the bottleneck for dataset
    # generation speed.
    self._inconsistency_by_known_production = {}
    logging.info(
        'Inference engine updated: %d monotonic productions, %d all '
        'productions.', len(self.monotonic_productions),
        len(self.all_productions))

  def force_add_production(
      self,
      production,
      is_monotonic = False,
  ):
    """Returns the status of adding a production to the inference engine.

    This method does not restore the state of the inference engine regardless
    of whether an inconsistency was detected or not.
    Args:
      production: The production to be added to the inference engine.
      is_monotonic: Whether the production should be added to the
        monotonic_productions collection (i.e., whether the production is being
        asserted as monotonically true).
    """
    status = self._add_production(production, is_monotonic, exhaustive=True)
    self._clear_inconsistency_by_known_production()

    return status

  def add_production(self,
                     production,
                     is_monotonic = False,
                     restore = False):
    """Adds the production to the inference engine.

    The current implementation updates the states as if the production has been
    added.  If this would cause any inconsistency, an InconsistencyError is
    raised with the inference engine's states unchanged.

    Args:
      production: The production to be added to the inference engine.
      is_monotonic: Whether the production should be added to the
        monotonic_productions collection (i.e., whether the production is being
        asserted as monotonically true).
      restore: If True, restore the inference engine's states regardless of
        whether an InconsistencyError is raised.  This option is used for doing
        a dry run of adding the production in order to detect inconsistency when
        we do not want to change the states.

    Raises:
      InconsistencyError: If adding the production would result in
        inconsistency.  If this happens the inference engine's states are
        restored.
    """
    backup = self.backup_states()
    status = self._add_production(
        production, is_monotonic=is_monotonic, exhaustive=False)
    if status.inconsistency:
      self._restore_states(backup)
      raise InconsistencyError(
          f'Adding production would cause inconsistency: '
          f'production: {production}, inconsistency: '
          f'{status.inconsistency.type}.')
    if restore:
      self._restore_states(backup)
    else:
      self._clear_inconsistency_by_known_production()

  def inconsistency_if_production_added(
      self, production):
    """Returns the inconsistency if the production was added.

    This method always restores the inference engine's states.

    Args:
      production: The production to be added monotonically to the inference
        engine.
    """
    if production in self._inconsistency_by_known_production:
      return self._inconsistency_by_known_production[production]

    backup = self.backup_states()
    status = self._add_production(
        production, is_monotonic=True, exhaustive=False)
    self._restore_states(backup)
    self._inconsistency_by_known_production[production] = status.inconsistency

    return status.inconsistency

  def contains_monotonic_production(self, production):
    return production in self.monotonic_productions

  def contains_defeasible_production(self, production):
    return self._production_is_defeasible(production)

  def contains_production(self, production):
    return production in self.all_productions
