# coding=utf-8
# Copyright 2022 The Conceptual Learning Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Simple utilities encapsulating default directory names for dataset output.

"""

import datetime
import os


def get_default_benchmark_suite_directory(
    benchmark_base_dir, generation_run_start_time):
  """Returns the full path to the benchmark suite directory to use by default.

  For use in cases where a benchmark suite directory is not explicitly specified
  on the generate_benchmark_suite command line.

  Args:
    benchmark_base_dir: Directory under which the benchmark suite directory will
      be created.
    generation_run_start_time: The time at which the generation run started. The
      benchmark suite directory will be named in 'YYYY-MM-DD-HHMMSS' format
      based on this timestamp.
  """
  return os.path.join(benchmark_base_dir,
                      generation_run_start_time.strftime('%Y-%m-%d-%H%M%S'))


def get_default_benchmark_directory(benchmark_suite_dir,
                                    dataset_spec_id,
                                    replica_index = 1):
  """Returns the full path to the benchmark directory to use by default.

  For use in cases where a benchmark suite directory is not explicitly specified
  on the generate_benchmark_suite command line.

  Args:
    benchmark_suite_dir: Directory under which the benchmark directory will be
      created.
    dataset_spec_id: The ID of the dataset spec from which the dataset will be
      generated.
    replica_index: 1-based index of the dataset, in the case where multiple
      datasets are to be generated from the same dataset spec.
  """
  return os.path.join(benchmark_suite_dir, f'{dataset_spec_id}_{replica_index}')
