# coding=utf-8
# Copyright 2022 The Conceptual Learning Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Definitions of enums that were factored out to avoid circular dependencies.

For example, if an enum was originally defined in `inputs.py`, but needs to be
referenced by a library that is imported by `conceptual_learning.py`, then the
simplest solution to avoid a circular dependency is to move the definition of
the enum into this file.

This file should contain just simple enum definitions, without any complex logic
that would lead to dependencies on other conceptual_learning libraries.
"""

import enum


@enum.unique
class RuleFormat(str, enum.Enum):
  """Supported formats for rule requests.

  During grammar generation, rules are initially represented as feature grammar
  productions, but when given to the learner as rule requests in conceptual
  learning examples, they are converted to strings in one of the formats
  described here.

  Example:
    Source rule:
      V[sem=(?x2+?x2+?x1)] -> U[sem=?x1] 'opposite' W[sem=?x2]
    FEATURE_GRAMMAR_PRODUCTION:
      "V[sem=(?x2+?x2+?x1)] -> U[sem=?x1] 'opposite' W[sem=?x2]"
    INTERPRETATION_RULE:
      "[x1 opposite x2] = [x2] [x2] [x1]"
    NATURAL_LANGUAGE:
      'the interpretation of a phrase x1 followed by "opposite" followed by a
       phrase x2 is the interpretation of x2 repeated twice followed by the
       interpretation of x1'
  """
  FEATURE_GRAMMAR_PRODUCTION = 'FEATURE_GRAMMAR_PRODUCTION'
  INTERPRETATION_RULE = 'INTERPRETATION_RULE'
  NATURAL_LANGUAGE = 'NATURAL_LANGUAGE'
