# coding=utf-8
# Copyright 2022 The Conceptual Learning Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Library for retrieving canned SCAN or SCAN-like grammars, e.g., for testing.
"""

import enum
import nltk



@enum.unique
class StandardGrammarId(enum.Enum):
  """Identifier of standard grammars available via load_standard_grammar().

  The string value is by construction equal to the filename in the data
  director.
  """
  SCAN_FINITE = 'scan_finite'
  SCAN_FINITE_NYE = 'scan_finite_nye'
  SCAN_INFINITE = 'scan_infinite'
  SCAN_INFINITE_NYE = 'scan_infinite_nye'
  SCAN_FINITE_NYE_STANDARDIZED = 'scan_finite_nye_standardized'
  SCAN_FINITE_12_RULES = 'scan_finite_12_rules'
  SCAN_EXTENDED = 'scan_extended'


def load_standard_grammar_string(grammar_id):
  """Returns the specified standard grammar in string format.

  Args:
    grammar_id: The grammar to retrieve.

  Returns:
    The grammar in a string format suitable for parsing into a FeatureGrammar.
  """

  raise NotImplementedError(
      f'TODO: Insert system-specific code to load the contents of the file '
      f'"conceptual_learning/cscan/data/{grammar_id.value}.fcfg".')


def load_standard_grammar(
    grammar_id):
  """Returns the specified standard grammar."""
  grammar_string = load_standard_grammar_string(grammar_id)
  grammar = nltk.grammar.FeatureGrammar.fromstring(grammar_string)
  return grammar
