# coding=utf-8
# Copyright 2022 The Conceptual Learning Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from absl.testing import absltest
from absl.testing import parameterized

from conceptual_learning.cscan import grammar_loader
from conceptual_learning.cscan import grammar_representation
from conceptual_learning.cscan import test_utils


class GrammarLoaderTest(parameterized.TestCase):

  def test_load_standard_grammar(self):
    # Verify that all of the standard grammars can be successfully loaded.
    for grammar_id in grammar_loader.StandardGrammarId:
      grammar_string = grammar_loader.load_standard_grammar_string(grammar_id)
      self.assertNotEmpty(grammar_string)
      grammar = grammar_loader.load_standard_grammar(grammar_id)
      self.assertIsNotNone(grammar)

  @parameterized.named_parameters(
      ('SCAN_FINITE_NYE_STANDARDIZED',
       grammar_loader.StandardGrammarId.SCAN_FINITE_NYE_STANDARDIZED),
      ('SCAN_EXTENDED', grammar_loader.StandardGrammarId.SCAN_EXTENDED),
  )
  def test_roundtrip_important_grammars(self, grammar_id):
    # Check that the serialization round trip is the identity.
    grammar = grammar_loader.load_standard_grammar(grammar_id)
    self.assertIsNotNone(grammar)
    grammar_schema = (
        grammar_representation.grammar_schema_from_feature_grammar(grammar))
    actual_grammar_string = grammar_schema.to_grammar_string()
    expected_grammar_string = test_utils.strip_blank_and_comment_lines(
        grammar_loader.load_standard_grammar_string(grammar_id))
    self.assertEqual(expected_grammar_string, actual_grammar_string)


if __name__ == '__main__':
  absltest.main()
