# coding=utf-8
# Copyright 2022 The Conceptual Learning Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import os

from absl.testing import absltest
import tensorflow as tf

from conceptual_learning.util import io_utils


class IoUtilTest(absltest.TestCase):

  def test_back_up_existing_file_noop_if_file_does_not_exist(self):
    parent_dir = self.create_tempdir()
    filename = 'filename.txt'
    filepath = os.path.join(parent_dir, filename)
    io_utils.back_up_existing_file(filepath)
    parent_dir_content = tf.io.gfile.listdir(parent_dir)
    self.assertEmpty(parent_dir_content)

  def test_back_up_existing_file_appends_index_if_file_exist(self):
    parent_dir = self.create_tempdir()
    filename = 'filename.txt'
    filepath = os.path.join(parent_dir, filename)

    # Write the file the first time.
    with tf.io.gfile.GFile(filepath, 'w') as f:
      f.write('Some content.')

    # Back up first time.
    io_utils.back_up_existing_file(filepath)
    parent_dir_content_1 = tf.io.gfile.listdir(parent_dir)

    # Write the file the second time.
    with tf.io.gfile.GFile(filepath, 'w') as f:
      f.write('Some new content.')

    # Back up second time.
    io_utils.back_up_existing_file(filepath)
    parent_dir_content_2 = tf.io.gfile.listdir(parent_dir)

    with self.subTest('appends_1_when_backing_up_the_first_time'):
      self.assertListEqual(parent_dir_content_1, ['filename_1.txt'])

    with self.subTest('increases_index_when_backing_up_subsequent_times'):
      self.assertListEqual(parent_dir_content_2,
                           ['filename_1.txt', 'filename_2.txt'])


if __name__ == '__main__':
  absltest.main()
