import time

import numpy as np
import pybullet as pb
import pybullet_data

import sys

from pathlib import Path
BASEDIR = Path(__file__).parent.parent
sys.path.insert(0, str(BASEDIR))

from modules import sim
from modules.sim import asset


def random_shelf_scene(env_offset, asset_util, obj_scale_up=2, shelf_scale_up=3, gui=False):
    pb.connect(pb.GUI if gui else pb.DIRECT)
    pb.setAdditionalSearchPath(pybullet_data.getDataPath())
    pb.loadURDF("plane.urdf")
    pb.setGravity(0, 0, -9.8)


    # Overview

    # world_frame
    #   - shelf_pose_randomizer
    #       - shelf_frame
    #           - box0
    #           - ...
    #           - box4
    #           - shelf_region
    #           - pose_randomizer
    #               - obj0
    #               - ...
    #               - obj3

    # Create the world frame
    world_frame = sim.SimObject(name="world")

    # Create the shelf_pose_randomizer
    shelf_pose_randomizer = sim.PoseRandomizer(
        name="shelf_pose_randomizer",
        position_min=np.array([-0.1, -0.1, 0]) + env_offset,
        position_max=np.array([0.1, 0.1, 0.1]) + env_offset,
        euler_angle_min=np.array([0, 0, np.pi]),
        euler_angle_max=np.array([0, 0, np.pi]),
    )
    world_frame.add_child_object(shelf_pose_randomizer)

    # Create the shelf frame
    shelf_frame = sim.SimObject(name="shelf_frame")
    shelf_pose_randomizer.add_child_object(shelf_frame)

    # Create the shelf
    lx = 0.16 * shelf_scale_up
    ly = 0.16 * shelf_scale_up
    lz = 0.14 * shelf_scale_up
    ceil_offset = -0.1

    positions = [
        [0, 0, 0],
        [0, 0, lz+ceil_offset],
        [0, ly / 2, lz / 2],
        [0, -ly / 2, lz / 2],
        [-lx / 2, 0, lz / 2],
    ]

    valid_AABB = np.array([-lx / 2, -ly / 2, 0]), np.array([lx / 2, ly / 2, lz+ceil_offset])
    valid_AABB = (valid_AABB[0]+env_offset, valid_AABB[1]+env_offset)

    # for i in range(5):
    for i, shelf_mesh_dir in enumerate(["shelf_gen/cvx/shelf_com0.obj", "shelf_gen/cvx/shelf_com1.obj", "shelf_gen/cvx/shelf_com2.obj", "shelf_gen/cvx/shelf_com3.obj", "shelf_gen/cvx/shelf_com4.obj"]):
        visual_shape_id = pb.createVisualShape(pb.GEOM_MESH,
                                               fileName=asset_util.get_obj_path_from_rel_path(shelf_mesh_dir),
                                               meshScale=[shelf_scale_up] * 3)
        collision_shape_id = pb.createCollisionShape(
            pb.GEOM_MESH,
            fileName=asset_util.get_obj_path_from_rel_path(shelf_mesh_dir),
            meshScale=[shelf_scale_up] * 3)
        box = sim.RigidObject(
            name=f"box{i}",
            visual_shape_id=visual_shape_id,
            collision_shape_id=collision_shape_id,
            position=positions[i],
        )
        shelf_frame.add_child_object(box)
    
    # Create the shelf region
    shelf_region = sim.BoxRegion(
        name="shelf_region",
        position=shelf_frame.convert_position_to_world_frame(
            np.array([0, 0, 0])),
        orientation=shelf_frame.convert_orientation_to_world_frame(
            pb.getQuaternionFromEuler([0, 0, 0])),
        min_position=np.array([-lx / 2, -ly / 2, 0]),
        max_position=np.array([lx / 2, ly / 2, lz]))
    shelf_frame.add_child_object(shelf_region)

    # Create the pose randomizer
    pose_randomizer = sim.PoseRandomizer(
        name="pose_randomizer",
        position=shelf_frame.convert_position_to_world_frame(
            np.array([0, 0, 0])),
        orientation=shelf_frame.convert_orientation_to_world_frame(
            pb.getQuaternionFromEuler([0, 0, 0])),
        position_min=np.array([-lx / 2, -ly / 2, 0]),
        position_max=np.array([lx / 2 - 0.03, ly / 2, lz / 2]),
        euler_angle_min=np.array([np.pi / 2, 0, 0]),
        euler_angle_max=np.array([np.pi / 2, 0, 0]),
    )
    shelf_frame.add_child_object(pose_randomizer)


    # Add a can object to the pose randomizer
    # can_path = asset.get_asset_path("can.obj")
    can_path = asset_util.get_obj_path_from_rel_path("NOCS/modified/can-d3e24e7712e1e82dece466fd8a3f2b40.obj")

    scale = 0.1 * obj_scale_up
    for i in range(4):
        visual_shape_id = pb.createVisualShape(
            pb.GEOM_MESH,
            fileName=can_path,
            meshScale=[scale] * 3,
        )
        collision_shape_id = pb.createCollisionShape(
            pb.GEOM_MESH,
            fileName=can_path,
            meshScale=[scale] * 3,
        )
        obj = sim.RigidObject(
            name=f"obj{i}",
            visual_shape_id=visual_shape_id,
            collision_shape_id=collision_shape_id,
            mass=0.1,
        )
        pose_randomizer.add_child_object(obj)

    condition_function = lambda obj: not obj.is_collision()

    shelf_pose_randomizer.randomize_pose(
        shelf_pose_randomizer.child_objects.values())
    pose_randomizer.randomize_pose(
        pose_randomizer.child_objects.values(),
        condition_function=condition_function)
    
    # Visualize the shelf region and the pose randomizer
    # Comment out the following two lines to speed up the simulation
    if gui:
        pb.removeAllUserDebugItems()
        shelf_region.visualize()
        pose_randomizer.visualize()

    # Run the simulation
    pb.setTimeStep(1 / 240)
    for i in range(2*240):
        pb.stepSimulation()

    fixed_obj_uid = [shelf_pose_randomizer.child_objects['shelf_frame'].child_objects[f'box{i}'].id for i in range(5)]
    movable_obj_uid = [pose_randomizer.child_objects[k].id for k in pose_randomizer.child_objects]

    return movable_obj_uid, fixed_obj_uid, valid_AABB



if __name__ == "__main__":
    pb.connect(pb.GUI)
    pb.setAdditionalSearchPath(pybullet_data.getDataPath())
    pb.loadURDF("plane.urdf")
    pb.setGravity(0, 0, -9.8)


    # Overview

    # world_frame
    #   - shelf_pose_randomizer
    #       - shelf_frame
    #           - box0
    #           - ...
    #           - box4
    #           - shelf_region
    #           - pose_randomizer
    #               - obj0
    #               - ...
    #               - obj3

    # Create the world frame
    world_frame = sim.SimObject(name="world")

    # Create the shelf_pose_randomizer
    shelf_pose_randomizer = sim.PoseRandomizer(
        name="shelf_pose_randomizer",
        position_min=np.array([-0.1, -0.1, 0]),
        position_max=np.array([0.1, 0.1, 0.1]),
        euler_angle_min=np.array([0, 0, - np.pi / 8]),
        euler_angle_max=np.array([0, 0, np.pi / 8]),
    )
    world_frame.add_child_object(shelf_pose_randomizer)

    # Create the shelf frame
    shelf_frame = sim.SimObject(name="shelf_frame")
    shelf_pose_randomizer.add_child_object(shelf_frame)

    # Create the shelf
    lx = 0.16
    ly = 0.16
    lz = 0.14
    d = 0.01

    half_extents = [
        [lx / 2, ly / 2, d],
        [lx / 2, ly / 2, d],
        [lx / 2, d, lz / 2],
        [lx / 2, d, lz / 2],
        [d, ly / 2, lz / 2],
    ]
    positions = [
        [0, 0, 0],
        [0, 0, lz],
        [0, ly / 2, lz / 2],
        [0, -ly / 2, lz / 2],
        [-lx / 2, 0, lz / 2],
    ]

    for i in range(5):
        visual_shape_id = pb.createVisualShape(pb.GEOM_BOX,
                                               halfExtents=half_extents[i])
        collision_shape_id = pb.createCollisionShape(
            pb.GEOM_BOX, halfExtents=half_extents[i])
        box = sim.RigidObject(
            name=f"box{i}",
            visual_shape_id=visual_shape_id,
            collision_shape_id=collision_shape_id,
            position=positions[i],
        )
        shelf_frame.add_child_object(box)
    
    # Create the shelf region
    shelf_region = sim.BoxRegion(
        name="shelf_region",
        position=shelf_frame.convert_position_to_world_frame(
            np.array([0, 0, 0])),
        orientation=shelf_frame.convert_orientation_to_world_frame(
            pb.getQuaternionFromEuler([0, 0, 0])),
        min_position=np.array([-lx / 2, -ly / 2, 0]),
        max_position=np.array([lx / 2, ly / 2, lz]))
    shelf_frame.add_child_object(shelf_region)

    # Create the pose randomizer
    pose_randomizer = sim.PoseRandomizer(
        name="pose_randomizer",
        position=shelf_frame.convert_position_to_world_frame(
            np.array([0, 0, 0])),
        orientation=shelf_frame.convert_orientation_to_world_frame(
            pb.getQuaternionFromEuler([0, 0, 0])),
        position_min=np.array([-lx / 2, -ly / 2, 0]),
        position_max=np.array([lx / 2 - 0.03, ly / 2, lz / 2]),
        euler_angle_min=np.array([np.pi / 2, 0, 0]),
        euler_angle_max=np.array([np.pi / 2, 0, 0]),
    )
    shelf_frame.add_child_object(pose_randomizer)


    # Add a can object to the pose randomizer
    can_path = asset.get_asset_path("can.obj")
    scale = 0.1
    for i in range(4):
        visual_shape_id = pb.createVisualShape(
            pb.GEOM_MESH,
            fileName=can_path,
            meshScale=[scale] * 3,
        )
        collision_shape_id = pb.createCollisionShape(
            pb.GEOM_MESH,
            fileName=can_path,
            meshScale=[scale] * 3,
        )
        obj = sim.RigidObject(
            name=f"obj{i}",
            visual_shape_id=visual_shape_id,
            collision_shape_id=collision_shape_id,
            mass=0.1,
        )
        pose_randomizer.add_child_object(obj)

    condition_function = lambda obj: not obj.is_collision()

    # Run the simulation
    pb.setTimeStep(1 / 240)
    for i in range(240 * 100):
        pb.stepSimulation()
        time.sleep(1 / 240)
        if i % 240 == 0:
            # Randomize the shelf region and the pose randomizer
            shelf_pose_randomizer.randomize_pose(
                shelf_pose_randomizer.child_objects.values())
            pose_randomizer.randomize_pose(
                pose_randomizer.child_objects.values(),
                condition_function=condition_function)
            
            # Visualize the shelf region and the pose randomizer
            # Comment out the following two lines to speed up the simulation
            pb.removeAllUserDebugItems()
            shelf_region.visualize()
            pose_randomizer.visualize()

    pb.disconnect()
