import numpy as np
import pybullet as pb

from .sim_object import *

class Region(SimObject):
    """Class for a region in the environment.
    """

    def __init__(
        self,
        name: str,
        position: np.ndarray = np.array([0, 0, 0]),
        orientation: np.ndarray = np.array([0, 0, 0, 1]),
        scale: float = 1,
    ) -> None:
        super().__init__(name, position, orientation, scale)


class BoxRegion(Region):
    """Class for a box region in the environment.
    """

    def __init__(
        self,
        name: str,
        position: np.ndarray = np.array([0, 0, 0]),
        orientation: np.ndarray = np.array([0, 0, 0, 1]),
        scale: float = 1,
        min_position: np.ndarray = np.array([0, 0, 0]),
        max_position: np.ndarray = np.array([0, 0, 0]),
    ) -> None:
        super().__init__(name, position, orientation, scale)

        self.min_position = min_position
        self.max_position = max_position

    def in_region(self, object: RigidObject) -> bool:
        """Checks if the object is within the box region."""

        local_position = self.convert_position_to_local_frame(object.get_position())
        return np.all(local_position >= self.min_position) and np.all(local_position <= self.max_position)

    def visualize(
            self,
            rgb_color: np.ndarray = np.array([0, 1, 1]),
    ) -> None:
        """Draws the box region in the PyBullet client."""

        super().visualize()

        corner_points = [
            self.min_position,
            np.array([self.min_position[0], self.min_position[1], self.max_position[2]]),
            np.array([self.min_position[0], self.max_position[1], self.min_position[2]]),
            np.array([self.min_position[0], self.max_position[1], self.max_position[2]]),
            np.array([self.max_position[0], self.min_position[1], self.min_position[2]]),
            np.array([self.max_position[0], self.min_position[1], self.max_position[2]]),
            np.array([self.max_position[0], self.max_position[1], self.min_position[2]]),
            self.max_position,
        ]

        world_corner_points = [self.convert_position_to_world_frame(point) for point in corner_points]

        for start, end in [
            (0, 1), (0, 2), (0, 4), (1, 3), (1, 5), (2, 3), (2, 6), (3, 7),
            (4, 5), (4, 6), (5, 7), (6, 7)
        ]:
            pb.addUserDebugLine(
            world_corner_points[start],
            world_corner_points[end],
            lineColorRGB=rgb_color,
            lifeTime=0,
            lineWidth=2
            )