import os
import logging
from typing import Tuple, List, Optional

import numpy as np
import torchvision.transforms as transforms
import torch
import torch.nn.functional as F
from torch.utils.data import Dataset
from PIL import Image
from torchvision.transforms.functional import InterpolationMode

from datasets.utils import set_default_from_args
from utils.conf import base_path
from datasets.utils.continual_dataset import (
    ContinualDataset,
    fix_class_names_order,
    store_masked_loaders,
)
from datasets.transforms.denormalization import DeNormalize

num_task = 1  # Number of tasks to split ImageNet-1K into

IMAGENET1K_CLASSNAMES: Optional[List[str]] = [
            "tench", "goldfish", "great white shark", "tiger shark", "hammerhead shark", "electric ray",
            "stingray", "rooster", "hen", "ostrich", "brambling", "goldfinch", "house finch", "junco",
            "indigo bunting", "American robin", "bulbul", "jay", "magpie", "chickadee", "American dipper",
            "kite (bird of prey)", "bald eagle", "vulture", "great grey owl", "fire salamander",
            "smooth newt", "newt", "spotted salamander", "axolotl", "American bullfrog", "tree frog",
            "tailed frog", "loggerhead sea turtle", "leatherback sea turtle", "mud turtle", "terrapin",
            "box turtle", "banded gecko", "green iguana", "Carolina anole",
            "desert grassland whiptail lizard", "agama", "frilled-necked lizard", "alligator lizard",
            "Gila monster", "European green lizard", "chameleon", "Komodo dragon", "Nile crocodile",
            "American alligator", "triceratops", "worm snake", "ring-necked snake",
            "eastern hog-nosed snake", "smooth green snake", "kingsnake", "garter snake", "water snake",
            "vine snake", "night snake", "boa constrictor", "African rock python", "Indian cobra",
            "green mamba", "sea snake", "Saharan horned viper", "eastern diamondback rattlesnake",
            "sidewinder rattlesnake", "trilobite", "harvestman", "scorpion", "yellow garden spider",
            "barn spider", "European garden spider", "southern black widow", "tarantula", "wolf spider",
            "tick", "centipede", "black grouse", "ptarmigan", "ruffed grouse", "prairie grouse", "peafowl",
            "quail", "partridge", "african grey parrot", "macaw", "sulphur-crested cockatoo", "lorikeet",
            "coucal", "bee eater", "hornbill", "hummingbird", "jacamar", "toucan", "duck",
            "red-breasted merganser", "goose", "black swan", "tusker", "echidna", "platypus", "wallaby",
            "koala", "wombat", "jellyfish", "sea anemone", "brain coral", "flatworm", "nematode", "conch",
            "snail", "slug", "sea slug", "chiton", "chambered nautilus", "Dungeness crab", "rock crab",
            "fiddler crab", "red king crab", "American lobster", "spiny lobster", "crayfish", "hermit crab",
            "isopod", "white stork", "black stork", "spoonbill", "flamingo", "little blue heron",
            "great egret", "bittern bird", "crane bird", "limpkin", "common gallinule", "American coot",
            "bustard", "ruddy turnstone", "dunlin", "common redshank", "dowitcher", "oystercatcher",
            "pelican", "king penguin", "albatross", "grey whale", "killer whale", "dugong", "sea lion",
            "Chihuahua", "Japanese Chin", "Maltese", "Pekingese", "Shih Tzu", "King Charles Spaniel",
            "Papillon", "toy terrier", "Rhodesian Ridgeback", "Afghan Hound", "Basset Hound", "Beagle",
            "Bloodhound", "Bluetick Coonhound", "Black and Tan Coonhound", "Treeing Walker Coonhound",
            "English foxhound", "Redbone Coonhound", "borzoi", "Irish Wolfhound", "Italian Greyhound",
            "Whippet", "Ibizan Hound", "Norwegian Elkhound", "Otterhound", "Saluki", "Scottish Deerhound",
            "Weimaraner", "Staffordshire Bull Terrier", "American Staffordshire Terrier",
            "Bedlington Terrier", "Border Terrier", "Kerry Blue Terrier", "Irish Terrier",
            "Norfolk Terrier", "Norwich Terrier", "Yorkshire Terrier", "Wire Fox Terrier",
            "Lakeland Terrier", "Sealyham Terrier", "Airedale Terrier", "Cairn Terrier",
            "Australian Terrier", "Dandie Dinmont Terrier", "Boston Terrier", "Miniature Schnauzer",
            "Giant Schnauzer", "Standard Schnauzer", "Scottish Terrier", "Tibetan Terrier",
            "Australian Silky Terrier", "Soft-coated Wheaten Terrier", "West Highland White Terrier",
            "Lhasa Apso", "Flat-Coated Retriever", "Curly-coated Retriever", "Golden Retriever",
            "Labrador Retriever", "Chesapeake Bay Retriever", "German Shorthaired Pointer", "Vizsla",
            "English Setter", "Irish Setter", "Gordon Setter", "Brittany dog", "Clumber Spaniel",
            "English Springer Spaniel", "Welsh Springer Spaniel", "Cocker Spaniel", "Sussex Spaniel",
            "Irish Water Spaniel", "Kuvasz", "Schipperke", "Groenendael dog", "Malinois", "Briard",
            "Australian Kelpie", "Komondor", "Old English Sheepdog", "Shetland Sheepdog", "collie",
            "Border Collie", "Bouvier des Flandres dog", "Rottweiler", "German Shepherd Dog", "Dobermann",
            "Miniature Pinscher", "Greater Swiss Mountain Dog", "Bernese Mountain Dog",
            "Appenzeller Sennenhund", "Entlebucher Sennenhund", "Boxer", "Bullmastiff", "Tibetan Mastiff",
            "French Bulldog", "Great Dane", "St. Bernard", "husky", "Alaskan Malamute", "Siberian Husky",
            "Dalmatian", "Affenpinscher", "Basenji", "pug", "Leonberger", "Newfoundland dog",
            "Great Pyrenees dog", "Samoyed", "Pomeranian", "Chow Chow", "Keeshond", "brussels griffon",
            "Pembroke Welsh Corgi", "Cardigan Welsh Corgi", "Toy Poodle", "Miniature Poodle",
            "Standard Poodle", "Mexican hairless dog (xoloitzcuintli)", "grey wolf", "Alaskan tundra wolf",
            "red wolf or maned wolf", "coyote", "dingo", "dhole", "African wild dog", "hyena", "red fox",
            "kit fox", "Arctic fox", "grey fox", "tabby cat", "tiger cat", "Persian cat", "Siamese cat",
            "Egyptian Mau", "cougar", "lynx", "leopard", "snow leopard", "jaguar", "lion", "tiger",
            "cheetah", "brown bear", "American black bear", "polar bear", "sloth bear", "mongoose",
            "meerkat", "tiger beetle", "ladybug", "ground beetle", "longhorn beetle", "leaf beetle",
            "dung beetle", "rhinoceros beetle", "weevil", "fly", "bee", "ant", "grasshopper",
            "cricket insect", "stick insect", "cockroach", "praying mantis", "cicada", "leafhopper",
            "lacewing", "dragonfly", "damselfly", "red admiral butterfly", "ringlet butterfly",
            "monarch butterfly", "small white butterfly", "sulphur butterfly", "gossamer-winged butterfly",
            "starfish", "sea urchin", "sea cucumber", "cottontail rabbit", "hare", "Angora rabbit",
            "hamster", "porcupine", "fox squirrel", "marmot", "beaver", "guinea pig", "common sorrel horse",
            "zebra", "pig", "wild boar", "warthog", "hippopotamus", "ox", "water buffalo", "bison",
            "ram (adult male sheep)", "bighorn sheep", "Alpine ibex", "hartebeest", "impala (antelope)",
            "gazelle", "arabian camel", "llama", "weasel", "mink", "European polecat",
            "black-footed ferret", "otter", "skunk", "badger", "armadillo", "three-toed sloth", "orangutan",
            "gorilla", "chimpanzee", "gibbon", "siamang", "guenon", "patas monkey", "baboon", "macaque",
            "langur", "black-and-white colobus", "proboscis monkey", "marmoset", "white-headed capuchin",
            "howler monkey", "titi monkey", "Geoffroy's spider monkey", "common squirrel monkey",
            "ring-tailed lemur", "indri", "Asian elephant", "African bush elephant", "red panda",
            "giant panda", "snoek fish", "eel", "silver salmon", "rock beauty fish", "clownfish",
            "sturgeon", "gar fish", "lionfish", "pufferfish", "abacus", "abaya", "academic gown",
            "accordion", "acoustic guitar", "aircraft carrier", "airliner", "airship", "altar", "ambulance",
            "amphibious vehicle", "analog clock", "apiary", "apron", "trash can", "assault rifle",
            "backpack", "bakery", "balance beam", "balloon", "ballpoint pen", "Band-Aid", "banjo",
            "baluster / handrail", "barbell", "barber chair", "barbershop", "barn", "barometer", "barrel",
            "wheelbarrow", "baseball", "basketball", "bassinet", "bassoon", "swimming cap", "bath towel",
            "bathtub", "station wagon", "lighthouse", "beaker", "military hat (bearskin or shako)",
            "beer bottle", "beer glass", "bell tower", "baby bib", "tandem bicycle", "bikini",
            "ring binder", "binoculars", "birdhouse", "boathouse", "bobsleigh", "bolo tie", "poke bonnet",
            "bookcase", "bookstore", "bottle cap", "hunting bow", "bow tie", "brass memorial plaque", "bra",
            "breakwater", "breastplate", "broom", "bucket", "buckle", "bulletproof vest",
            "high-speed train", "butcher shop", "taxicab", "cauldron", "candle", "cannon", "canoe",
            "can opener", "cardigan", "car mirror", "carousel", "tool kit", "cardboard box / carton",
            "car wheel", "automated teller machine", "cassette", "cassette player", "castle", "catamaran",
            "CD player", "cello", "mobile phone", "chain", "chain-link fence", "chain mail", "chainsaw",
            "storage chest", "chiffonier", "bell or wind chime", "china cabinet", "Christmas stocking",
            "church", "movie theater", "cleaver", "cliff dwelling", "cloak", "clogs", "cocktail shaker",
            "coffee mug", "coffeemaker", "spiral or coil", "combination lock", "computer keyboard",
            "candy store", "container ship", "convertible", "corkscrew", "cornet", "cowboy boot",
            "cowboy hat", "cradle", "construction crane", "crash helmet", "crate", "infant bed",
            "Crock Pot", "croquet ball", "crutch", "cuirass", "dam", "desk", "desktop computer",
            "rotary dial telephone", "diaper", "digital clock", "digital watch", "dining table",
            "dishcloth", "dishwasher", "disc brake", "dock", "dog sled", "dome", "doormat", "drilling rig",
            "drum", "drumstick", "dumbbell", "Dutch oven", "electric fan", "electric guitar",
            "electric locomotive", "entertainment center", "envelope", "espresso machine", "face powder",
            "feather boa", "filing cabinet", "fireboat", "fire truck", "fire screen", "flagpole", "flute",
            "folding chair", "football helmet", "forklift", "fountain", "fountain pen", "four-poster bed",
            "freight car", "French horn", "frying pan", "fur coat", "garbage truck",
            "gas mask or respirator", "gas pump", "goblet", "go-kart", "golf ball", "golf cart", "gondola",
            "gong", "gown", "grand piano", "greenhouse", "radiator grille", "grocery store", "guillotine",
            "hair clip", "hair spray", "half-track", "hammer", "hamper", "hair dryer", "hand-held computer",
            "handkerchief", "hard disk drive", "harmonica", "harp", "combine harvester", "hatchet",
            "holster", "home theater", "honeycomb", "hook", "hoop skirt", "gymnastic horizontal bar",
            "horse-drawn vehicle", "hourglass", "iPod", "clothes iron", "carved pumpkin", "jeans", "jeep",
            "T-shirt", "jigsaw puzzle", "rickshaw", "joystick", "kimono", "knee pad", "knot", "lab coat",
            "ladle", "lampshade", "laptop computer", "lawn mower", "lens cap", "letter opener", "library",
            "lifeboat", "lighter", "limousine", "ocean liner", "lipstick", "slip-on shoe", "lotion",
            "music speaker", "loupe magnifying glass", "sawmill", "magnetic compass", "messenger bag",
            "mailbox", "tights", "one-piece bathing suit", "manhole cover", "maraca", "marimba", "mask",
            "matchstick", "maypole", "maze", "measuring cup", "medicine cabinet", "megalith", "microphone",
            "microwave oven", "military uniform", "milk can", "minibus", "miniskirt", "minivan", "missile",
            "mitten", "mixing bowl", "mobile home", "ford model t", "modem", "monastery", "monitor",
            "moped", "mortar and pestle", "graduation cap", "mosque", "mosquito net", "vespa",
            "mountain bike", "tent", "computer mouse", "mousetrap", "moving van", "muzzle", "metal nail",
            "neck brace", "necklace", "baby pacifier", "notebook computer", "obelisk", "oboe", "ocarina",
            "odometer", "oil filter", "pipe organ", "oscilloscope", "overskirt", "bullock cart",
            "oxygen mask", "product packet / packaging", "paddle", "paddle wheel", "padlock", "paintbrush",
            "pajamas", "palace", "pan flute", "paper towel", "parachute", "parallel bars", "park bench",
            "parking meter", "railroad car", "patio", "payphone", "pedestal", "pencil case",
            "pencil sharpener", "perfume", "Petri dish", "photocopier", "plectrum", "Pickelhaube",
            "picket fence", "pickup truck", "pier", "piggy bank", "pill bottle", "pillow", "ping-pong ball",
            "pinwheel", "pirate ship", "drink pitcher", "block plane", "planetarium", "plastic bag",
            "plate rack", "farm plow", "plunger", "Polaroid camera", "pole", "police van", "poncho",
            "pool table", "soda bottle", "plant pot", "potter's wheel", "power drill", "prayer rug",
            "printer", "prison", "missile", "projector", "hockey puck", "punching bag", "purse", "quill",
            "quilt", "race car", "racket", "radiator", "radio", "radio telescope", "rain barrel",
            "recreational vehicle", "fishing casting reel", "reflex camera", "refrigerator",
            "remote control", "restaurant", "revolver", "rifle", "rocking chair", "rotisserie", "eraser",
            "rugby ball", "ruler measuring stick", "sneaker", "safe", "safety pin", "salt shaker", "sandal",
            "sarong", "saxophone", "scabbard", "weighing scale", "school bus", "schooner", "scoreboard",
            "CRT monitor", "screw", "screwdriver", "seat belt", "sewing machine", "shield", "shoe store",
            "shoji screen / room divider", "shopping basket", "shopping cart", "shovel", "shower cap",
            "shower curtain", "ski", "balaclava ski mask", "sleeping bag", "slide rule", "sliding door",
            "slot machine", "snorkel", "snowmobile", "snowplow", "soap dispenser", "soccer ball", "sock",
            "solar thermal collector", "sombrero", "soup bowl", "keyboard space bar", "space heater",
            "space shuttle", "spatula", "motorboat", "spider web", "spindle", "sports car", "spotlight",
            "stage", "steam locomotive", "through arch bridge", "steel drum", "stethoscope", "scarf",
            "stone wall", "stopwatch", "stove", "strainer", "tram", "stretcher", "couch", "stupa",
            "submarine", "suit", "sundial", "sunglasses", "sunglasses", "sunscreen", "suspension bridge",
            "mop", "sweatshirt", "swim trunks / shorts", "swing", "electrical switch", "syringe",
            "table lamp", "tank", "tape player", "teapot", "teddy bear", "television", "tennis ball",
            "thatched roof", "front curtain", "thimble", "threshing machine", "throne", "tile roof",
            "toaster", "tobacco shop", "toilet seat", "torch", "totem pole", "tow truck", "toy store",
            "tractor", "semi-trailer truck", "tray", "trench coat", "tricycle", "trimaran", "tripod",
            "triumphal arch", "trolleybus", "trombone", "hot tub", "turnstile", "typewriter keyboard",
            "umbrella", "unicycle", "upright piano", "vacuum cleaner", "vase", "vaulted or arched ceiling",
            "velvet fabric", "vending machine", "vestment", "viaduct", "violin", "volleyball",
            "waffle iron", "wall clock", "wallet", "wardrobe", "military aircraft", "sink",
            "washing machine", "water bottle", "water jug", "water tower", "whiskey jug", "whistle",
            "hair wig", "window screen", "window shade", "Windsor tie", "wine bottle", "airplane wing",
            "wok", "wooden spoon", "wool", "split-rail fence", "shipwreck", "sailboat", "yurt", "website",
            "comic book", "crossword", "traffic or street sign", "traffic light", "dust jacket", "menu",
            "plate", "guacamole", "consomme", "hot pot", "trifle", "ice cream", "popsicle", "baguette",
            "bagel", "pretzel", "cheeseburger", "hot dog", "mashed potatoes", "cabbage", "broccoli",
            "cauliflower", "zucchini", "spaghetti squash", "acorn squash", "butternut squash", "cucumber",
            "artichoke", "bell pepper", "cardoon", "mushroom", "Granny Smith apple", "strawberry", "orange",
            "lemon", "fig", "pineapple", "banana", "jackfruit", "cherimoya (custard apple)", "pomegranate",
            "hay", "carbonara", "chocolate syrup", "dough", "meatloaf", "pizza", "pot pie", "burrito",
            "red wine", "espresso", "tea cup", "eggnog", "mountain", "bubble", "cliff", "coral reef",
            "geyser", "lakeshore", "promontory", "sandbar", "beach", "valley", "volcano", "baseball player",
            "bridegroom", "scuba diver", "rapeseed", "daisy", "yellow lady's slipper", "corn", "acorn",
            "rose hip", "horse chestnut seed", "coral fungus", "agaric", "gyromitra", "stinkhorn mushroom",
            "earth star fungus", "hen of the woods mushroom", "bolete", "corn cob", "toilet paper"
        ]
        
class MyImageNet1K(Dataset):
    N_CLASSES = 1000

    """
    Always use the 'val' subfolder as image source for both train and test.
    No download: the dataset must already exist under base_path()/imagenet1k/val.
    """

    def __init__(self, root, train=True, transform=None,
                 target_transform=None, download: bool = False) -> None:

    # We use the validation split for both cases
        self.root = os.path.join(root, 'val') if not str(root).endswith('val') else root
        self.train = train
        self.transform = transform
        self.target_transform = target_transform

    # No download: the dataset must already exist
        if not os.path.isdir(self.root):
            raise RuntimeError('Dataset not found. Expected at: ' + self.root)

        self.not_aug_transform = transforms.Compose([
            transforms.Resize((224, 224), interpolation=InterpolationMode.BICUBIC),
            transforms.ToTensor()
        ])

    # Prepare file list and targets by scanning folders (ImageFolder-style)
        valid_ext = {'.jpg', '.jpeg', '.png', '.ppm', '.bmp', '.pgm', '.tif', '.tiff', '.webp'}

        classes = [d for d in os.listdir(self.root)
                   if os.path.isdir(os.path.join(self.root, d))]
        classes.sort()
        self.class_to_idx = {c: i for i, c in enumerate(classes)}

        data, targets = [], []
        for cls in classes:
            cls_idx = self.class_to_idx[cls]
            cls_dir = os.path.join(self.root, cls)
            for dirpath, _, filenames in os.walk(cls_dir):
                for fname in filenames:
                    ext = os.path.splitext(fname)[1].lower()
                    if ext in valid_ext:
                        data.append(os.path.join(dirpath, fname))
                        targets.append(cls_idx)

        self.data = np.array(data)
        self.targets = np.array(targets)

    def __len__(self):
        return len(self.targets)

    def __getitem__(self, index: int) -> Tuple[Image.Image, int, Image.Image]:
        img_path, target = self.data[index], int(self.targets[index])

        img = Image.open(img_path).convert('RGB')
        original_img = img.copy()
        not_aug_img = self.not_aug_transform(original_img)

        if self.transform is not None:
            img = self.transform(img)

        if self.target_transform is not None:
            target = self.target_transform(target)

        if not self.train:
            return img, target

        if hasattr(self, 'logits'):
            return img, target, not_aug_img, self.logits[index]

        return img, target, not_aug_img


class SequentialImageNet1K(ContinualDataset):
    NAME = 'seq-imagenet1k'
    SETTING = 'class-il'
    N_TASKS = num_task
    N_CLASSES = 1000
    N_CLASSES_PER_TASK = N_CLASSES // N_TASKS
    MEAN, STD = (0.48145466, 0.4578275, 0.40821073), (0.26862954, 0.26130258, 0.27577711)
    SIZE = (224, 224)

    TRANSFORM = transforms.Compose([
        transforms.RandomResizedCrop(SIZE[0], interpolation=InterpolationMode.BICUBIC),
        transforms.RandomHorizontalFlip(),
        transforms.ToTensor(),
        transforms.Normalize(mean=MEAN, std=STD),
    ])
    TEST_TRANSFORM = transforms.Compose([
        transforms.Resize(size=(256, 256), interpolation=InterpolationMode.BICUBIC),
        transforms.CenterCrop(SIZE[0]),
        transforms.ToTensor(),
        transforms.Normalize(mean=MEAN, std=STD)
    ])

    def __init__(self, args):
    # Always force 4 tasks of 250 classes on 1000 total classes, regardless of external configs
        self.N_TASKS = num_task
        self.N_CLASSES = 1000
        self.N_CLASSES_PER_TASK = self.N_CLASSES // self.N_TASKS
        super().__init__(args)

    def get_data_loaders(self):
        root = os.path.join(base_path(), 'imagenet1k')
    # Log the active split parameters
        try:
            import logging as _logging
            _logging.info(f"[seq-imagenet1k] N_TASKS={self.N_TASKS} N_CLASSES_PER_TASK={self.N_CLASSES_PER_TASK} N_CLASSES={self.N_CLASSES} joint={getattr(self.args, 'joint', None)}")
        except Exception:
            pass

    # Always use the validation split for both train and test
        train_dataset = MyImageNet1K(root, train=True, transform=self.TRANSFORM)
        test_dataset = MyImageNet1K(root, train=False, transform=self.TEST_TRANSFORM)

        train, test = store_masked_loaders(train_dataset, test_dataset, self)

    # Diagnostics: verify that the per-task split is correct
        try:
            start_c, end_c = self.get_offsets()
            t_targets = train.dataset.targets
            te_targets = test.dataset.targets

            t_uniq = np.unique(t_targets) if isinstance(t_targets, np.ndarray) else torch.unique(t_targets).cpu().numpy()
            te_uniq = np.unique(te_targets) if isinstance(te_targets, np.ndarray) else torch.unique(te_targets).cpu().numpy()

            logging.info(f"[seq-imagenet1k] Task {self.c_task} offsets=({start_c},{end_c}) "
                         f"train_classes={len(t_uniq)} range=({t_uniq.min() if len(t_uniq)>0 else 'NA'},{t_uniq.max() if len(t_uniq)>0 else 'NA'}) "
                         f"n_train={len(t_targets)})")
            logging.info(f"[seq-imagenet1k] Task {self.c_task} test_classes={len(te_uniq)} "
                         f"range=({te_uniq.min() if len(te_uniq)>0 else 'NA'},{te_uniq.max() if len(te_uniq)>0 else 'NA'}) "
                         f"n_test={len(te_targets)})")

            if len(t_uniq) != self.N_CLASSES_PER_TASK:
                logging.warning(f"[seq-imagenet1k] Expected {self.N_CLASSES_PER_TASK} classes in train, found {len(t_uniq)}. "
                                f"joint={getattr(self.args, 'joint', None)} validation={getattr(self.args, 'validation', None)} validation_mode={getattr(self.args, 'validation_mode', None)}")
        except Exception as e:
            logging.error(f"[seq-imagenet1k] Task split diagnostics failed: {e}")

        return train, test

    def get_class_names(self):
        if self.class_names is not None:
            return self.class_names

    # 1) If provided, use the hand-written list
        if isinstance(IMAGENET1K_CLASSNAMES, list) and len(IMAGENET1K_CLASSNAMES) == self.N_CLASSES:
            classes = [x.replace('_', ' ') for x in IMAGENET1K_CLASSNAMES]
        else:
            # 2) Deduce from folder names in 'val'
            val_root = os.path.join(base_path(), 'imagenet1k', 'val')
            if not os.path.isdir(val_root):
                # fallback to root without 'val' (in case the split was already passed)
                val_root = os.path.join(base_path(), 'imagenet1k')

            classes = [d for d in os.listdir(val_root)
                       if os.path.isdir(os.path.join(val_root, d))]
            classes.sort()
            classes = [x.replace('_', ' ') for x in classes]

        classes = fix_class_names_order(classes, self.args)
        self.class_names = classes
        return self.class_names

    @staticmethod
    def get_transform():
        transform = transforms.Compose(
            [transforms.ToPILImage(), SequentialImageNet1K.TRANSFORM])
        return transform

    @set_default_from_args("backbone")
    def get_backbone(self):
        return "vit"

    @staticmethod
    def get_loss():
        return F.cross_entropy

    @staticmethod
    def get_normalization_transform():
        return transforms.Normalize(mean=SequentialImageNet1K.MEAN, std=SequentialImageNet1K.STD)

    @staticmethod
    def get_denormalization_transform():
        return DeNormalize(SequentialImageNet1K.MEAN, SequentialImageNet1K.STD)

    @set_default_from_args('n_epochs')
    def get_epochs(self):
        return 50

    @set_default_from_args('batch_size')
    def get_batch_size(self):
        return 128