import argparse
import torch
from torch.utils.data import DataLoader
import os
import json
import pandas as pd
import time
import collections
from humanfriendly import format_timespan

from advbench import datasets
from advbench import algorithms
from advbench import networks
from advbench import evalulation_methods
from advbench import hparams_registry
from advbench.lib import misc, meters, reporting

def main(args, hparams, test_hparams):

    device = 'cuda' if torch.cuda.is_available() else 'cpu'
    torch.manual_seed(0)

    # paths for saving output
    json_path = os.path.join(args.output_dir, 'results.json')
    ckpt_path = misc.stage_path(args.output_dir, 'ckpts')
    train_df_path = os.path.join(args.output_dir, 'train.pd')
    selection_df_path = os.path.join(args.output_dir, 'selection.pd')

    dataset = vars(datasets)[args.dataset](args.data_dir, device)

    classifier = vars(networks)[args.architecture](
        dataset.INPUT_SHAPE,
        dataset.NUM_CLASSES
    ).to(device)
    # classifier = torch.nn.DataParallel(
    #     classifier, 
    #     device_ids=[0, 1, 2, 3])
    # hparams['batch_size'] = hparams['batch_size'] * 4

    algorithm = vars(algorithms)[args.algorithm](
        classifier=classifier,
        hparams=hparams,
        device=device
    ).to(device)

    train_loader = DataLoader(
        dataset=dataset.splits['train'],
        batch_size=hparams['batch_size'],
        num_workers=dataset.N_WORKERS,
        pin_memory=False,
        shuffle=True,
        drop_last=algorithm.DROP_LAST_BATCH)
    validation_loader = DataLoader(
        dataset=dataset.splits['validation'],
        batch_size=test_hparams['batch_size'],
        num_workers=dataset.N_WORKERS,
        pin_memory=False,
        shuffle=False,
        drop_last=algorithm.DROP_LAST_BATCH)
    test_loader = DataLoader(
        dataset=dataset.splits['test'],
        batch_size=test_hparams['batch_size'],
        num_workers=dataset.N_WORKERS,
        pin_memory=False,
        shuffle=False,
        drop_last=algorithm.DROP_LAST_BATCH)

    def save_checkpoint(epoch):
        torch.save(
            obj={'state_dict': algorithm.state_dict()}, 
            f=os.path.join(ckpt_path, f'model_ckpt_{epoch}.pkl')
        )

    evaluators = [
        vars(evalulation_methods)[e](
            algorithm=algorithm,
            device=device,
            output_dir=args.output_dir,
            test_hparams=test_hparams)
        for e in args.evaluators]

    total_time = 0
    for epoch in range(0, dataset.N_EPOCHS):

        if dataset.HAS_LR_SCHEDULE:
            dataset.adjust_lr(algorithm.optimizer, epoch, hparams)

        if algorithm.HAS_HPARAM_UPDATE_SCHEDULE:
            algorithm.update_hparams(epoch)

        timer = meters.TimeMeter()
        epoch_start = time.time()
        for batch_idx, (imgs, labels) in enumerate(train_loader):

            timer.batch_start()
            if not dataset.ON_DEVICE:
                imgs, labels = imgs.to(device), labels.to(device)
            algorithm.step(imgs, labels)

            if algorithm.SAVE_CKPTS is True:
                if algorithm.ckpt_save_criterion() is True:
                    save_checkpoint(f'alg-{algorithm.ckpt_counter}')

            save_checkpoint(f'alg-{epoch}')

            if batch_idx % dataset.LOG_INTERVAL == 0:
                print(f'Epoch {epoch+1}/{dataset.N_EPOCHS} ', end='')
                print(f'[{batch_idx * imgs.size(0)}/{len(dataset.splits["train"])}', end=' ')
                print(f'({100. * batch_idx / len(train_loader):.0f}%)]\t', end='')
                for name, meter in algorithm.meters.items():
                    print(f'{name}: {meter.val:.3f} (avg. {meter.avg:.3f})\t', end='')
                print(f'Time: {timer.batch_time.val:.3f} (avg. {timer.batch_time.avg:.3f})')

            timer.batch_end()

        results = {'Epoch': epoch, 'Train': {}, 'Validation': {}, 'Test': {}}

        for name, meter in algorithm.meters.items():
            results['Train'].update({name: meter.avg})

        print('\nTrain')
        misc.print_row([key for key in results['Train'].keys()]) 
        misc.print_row([results['Train'][key] for key in results['Train'].keys()])

        for evaluator in evaluators:
            for k, v in evaluator.calculate(validation_loader).items():
                results['Validation'].update({k: v})

        print('\nValidation')
        misc.print_row([key for key in results['Validation'].keys()]) 
        misc.print_row([results['Validation'][key] for key in results['Validation'].keys()])

        for evaluator in evaluators:
            for k, v in evaluator.calculate(test_loader).items():
                results['Test'].update({k: v})

        print('\nTest')
        misc.print_row([key for key in results['Test'].keys()])
        misc.print_row([results['Test'][key] for key in results['Test'].keys()])

        epoch_time = time.time() - epoch_start
        total_time += epoch_time

        results.update({
            'Epoch-Time': epoch_time,
            'Total-Time': total_time})

        # print results
        print(f'Epoch: {epoch+1}/{dataset.N_EPOCHS}\t', end='')
        print(f'Epoch time: {format_timespan(epoch_time)}\t', end='')
        print(f'Total time: {format_timespan(total_time)}')

        results.update({'hparams': hparams, 'args': vars(args)})

        with open(json_path, 'a') as f:
            f.write(json.dumps(results, sort_keys=True) + '\n')

        if args.save_model_every_epoch is True:
            save_checkpoint(epoch)

        algorithm.reset_meters()        

    save_checkpoint('final')

    records = reporting.load_record(json_path)

    train_dict = collections.defaultdict(lambda: [])
    validation_dict = collections.defaultdict(lambda: [])
    test_dict = collections.defaultdict(lambda: [])

    for record in records:
        for k in records[0]['Train'].keys():
            train_dict[k].append(record['Train'][k])

        for k in records[0]['Validation'].keys():
            validation_dict[k].append(record['Validation'][k])
            test_dict[k].append(record['Test'][k])

    def dict_to_dataframe(split, d):
        df = pd.DataFrame.from_dict(d)
        df['Split'] = split
        df = df.join(pd.DataFrame({
            'Algorithm': args.algorithm,
            'trial_seed': args.trial_seed,
            'Architecture': args.architecture,
            'seed': args.seed,
            'path': args.output_dir
        }, index=df.index))
        df['Epoch'] = range(dataset.N_EPOCHS)
        return df

    train_df = dict_to_dataframe('Train', train_dict)
    validation_df = dict_to_dataframe('Validation', validation_dict)
    test_df = dict_to_dataframe('Test', test_dict)
    selection_df = pd.concat([validation_df, test_df], ignore_index=True)

    train_df.to_pickle(train_df_path)
    selection_df.to_pickle(selection_df_path)

    with open(os.path.join(args.output_dir, 'done'), 'w') as f:
        f.write('done')

if __name__ == '__main__':

    parser = argparse.ArgumentParser(description='Adversarial robustness')
    parser.add_argument('--data_dir', type=str, default='./advbench/data')
    parser.add_argument('--output_dir', type=str, default='train_output')
    parser.add_argument('--dataset', type=str, default='MNIST', help='Dataset to use')
    parser.add_argument('--algorithm', type=str, default='ERM', help='Algorithm to run')
    parser.add_argument('--hparams', type=str, help='JSON-serialized hparams dict')
    parser.add_argument('--hparams_seed', type=int, default=0, help='Seed for hyperparameters')
    parser.add_argument('--trial_seed', type=int, default=0, help='Trial number')
    parser.add_argument('--seed', type=int, default=0, help='Seed for everything else')
    parser.add_argument('--evaluators', type=str, nargs='+', default=['Clean'])
    parser.add_argument('--save_model_every_epoch', action='store_true')
    parser.add_argument('--architecture', type=str, default='LeNet', choices=networks.ARCHITECTURES, help='Classification architecture')
    args = parser.parse_args()

    os.makedirs(os.path.join(args.output_dir), exist_ok=True)

    print('Args:')
    for k, v in sorted(vars(args).items()):
        print(f'\t{k}: {v}')

    with open(os.path.join(args.output_dir, 'args.json'), 'w') as f:
        json.dump(args.__dict__, f, indent=2)

    if args.dataset not in vars(datasets):
        raise NotImplementedError(f'Dataset {args.dataset} is not implemented.')

    if args.hparams_seed == 0:
        hparams = hparams_registry.default_hparams(args.algorithm, args.dataset)
    else:
        seed = misc.seed_hash(args.hparams_seed, args.trial_seed)
        hparams = hparams_registry.random_hparams(args.algorithm, args.dataset, seed)
    if args.hparams:
        hparams.update(json.loads(args.hparams))

    print ('Hparams:')
    for k, v in sorted(hparams.items()):
        print(f'\t{k}: {v}')

    with open(os.path.join(args.output_dir, 'hparams.json'), 'w') as f:
        json.dump(hparams, f, indent=2)

    test_hparams = hparams_registry.test_hparams(args.algorithm, args.dataset)

    print('Test hparams:')
    for k, v in sorted(test_hparams.items()):
        print(f'\t{k}: {v}')

    with open(os.path.join(args.output_dir, 'test_hparams.json'), 'w') as f:
        json.dump(test_hparams, f, indent=2)

    main(args, hparams, test_hparams)