import os
import math
import wandb
import random
import logging
import inspect
import argparse
import datetime
import subprocess

from pathlib import Path
from tqdm.auto import tqdm
from einops import rearrange
from omegaconf import OmegaConf
from safetensors import safe_open
from typing import Dict, Optional, Tuple
import numpy as np
import re
from PIL import Image

import torch
import torchvision
import torchvision.transforms as transforms
import torch.nn.functional as F
import torch.distributed as dist
from torch.optim.swa_utils import AveragedModel
from torch.utils.data.distributed import DistributedSampler
from torch.nn.parallel import DistributedDataParallel as DDP

from diffusers.optimization import get_scheduler
from diffusers.utils import check_min_version
from diffusers.utils.import_utils import is_xformers_available

from animatediff.data.dataset import WebVid10M, HAAVideo, InternVidVideo
from animatediff.data.data_distort_utils import random_blur_whole_video, random_distort_whole_video, random_add_salt_and_pepper_noise_to_video, random_change_color, blur_whole_video, distort_whole_video, add_salt_and_pepper_noise_to_video, change_color, change_app_video_v4
from animatediff.utils.util import save_videos_grid, zero_rank_print

from animatediff.models.vae_decoder import DetailAutoencoderKL


def init_dist(launcher="slurm", backend='nccl', port=29500, **kwargs):
    """Initializes distributed environment."""
    if launcher == 'pytorch':
        rank = int(os.environ['RANK'])
        num_gpus = torch.cuda.device_count()
        local_rank = rank % num_gpus
        torch.cuda.set_device(local_rank)
        dist.init_process_group(backend=backend, **kwargs)
        
    elif launcher == 'slurm':
        proc_id = int(os.environ['SLURM_PROCID'])
        ntasks = int(os.environ['SLURM_NTASKS'])
        node_list = os.environ['SLURM_NODELIST']
        num_gpus = torch.cuda.device_count()
        local_rank = proc_id % num_gpus
        torch.cuda.set_device(local_rank)
        addr = subprocess.getoutput(
            f'scontrol show hostname {node_list} | head -n1')
        os.environ['MASTER_ADDR'] = addr
        os.environ['WORLD_SIZE'] = str(ntasks)
        os.environ['RANK'] = str(proc_id)
        port = os.environ.get('PORT', port)
        os.environ['MASTER_PORT'] = str(port)
        dist.init_process_group(backend=backend)
        zero_rank_print(f"proc_id: {proc_id}; local_rank: {local_rank}; ntasks: {ntasks}; node_list: {node_list}; num_gpus: {num_gpus}; addr: {addr}; port: {port}")
        
    else:
        raise NotImplementedError(f'Not implemented launcher type: `{launcher}`!')
    
    return local_rank



def main(    
    name: str,
    use_wandb: bool,
    launcher: str,
    
    output_dir: str,
    pretrained_model_path: str,

    train_data: Dict,
    validation_data: Dict,
    
    max_train_epoch: int = -1,
    max_train_steps: int = 100,
    validation_steps: int = 100,
    validation_steps_tuple: Tuple = (-1,),

    learning_rate: float = 3e-5,
    scale_lr: bool = False,
    lr_warmup_steps: int = 0,
    lr_scheduler: str = "constant",

    num_workers: int = 8,
    train_batch_size: int = 1,
    adam_beta1: float = 0.9,
    adam_beta2: float = 0.999,
    adam_weight_decay: float = 1e-2,
    adam_epsilon: float = 1e-08,
    max_grad_norm: float = 1.0,
    gradient_accumulation_steps: int = 1,
    gradient_checkpointing: bool = True,
    checkpointing_epochs: int = 5,
    checkpointing_steps: int = -1,
    checkpointing_steps_tuple: Tuple = (-1,),

    mixed_precision_training: bool = True,
    enable_xformers_memory_efficient_attention: bool = True,

    global_seed: int = 42,
    is_debug: bool = False,

    dataset: str = 'InternVid',

    cfg_app_change_video: float = 0.0,
    cfg_app_change_video_version: str = "1",
    cfg_app_change_video_type: str = "replace",

    distort_video_prob: float = 0.0,
    spatial_gradient_loss_weight: float = 0.0,
):
    check_min_version("0.10.0.dev0")

    # Initialize distributed training
    local_rank      = init_dist(launcher=launcher)
    global_rank     = dist.get_rank()
    num_processes   = dist.get_world_size()
    is_main_process = global_rank == 0

    seed = global_seed + global_rank
    torch.manual_seed(seed)
    
    # Logging folder
    folder_name = "debug" if is_debug else name + datetime.datetime.now().strftime("-%Y-%m-%dT%H-%M-%S")
    output_dir = os.path.join(output_dir, folder_name)
    if is_debug and os.path.exists(output_dir):
        os.system(f"rm -rf {output_dir}")

    *_, config = inspect.getargvalues(inspect.currentframe())

    # Make one log on every process with the configuration for debugging.
    logging.basicConfig(
        format="%(asctime)s - %(levelname)s - %(name)s - %(message)s",
        datefmt="%m/%d/%Y %H:%M:%S",
        level=logging.INFO,
    )

    if is_main_process and (not is_debug) and use_wandb:
        run = wandb.init(project="animatediff", name=folder_name, config=config)

    # Handle the output folder creation
    if is_main_process:
        os.makedirs(output_dir, exist_ok=True)
        os.makedirs(f"{output_dir}/samples", exist_ok=True)
        os.makedirs(f"{output_dir}/sanity_check", exist_ok=True)
        OmegaConf.save(config, os.path.join(output_dir, 'config.yaml'))

    # Load model.
    vae = DetailAutoencoderKL.from_pretrained(pretrained_model_path, subfolder="vae", low_cpu_mem_usage=False) # low_cpu_mem_usage=False since it has new parameters
    vae.enable_slicing()
    vae.eval()
    
    # Set vae trainable parameters
    vae.requires_grad_(False)
    for _, param in vae.decoder.channel_matching_convs.named_parameters():
        param.requires_grad = True
    for _, param in vae.decoder.ref_attn_blocks.named_parameters():
        param.requires_grad = True
    trainable_params = list(filter(lambda p: p.requires_grad, vae.parameters()))

    optimizer = torch.optim.AdamW(
        trainable_params,
        lr=learning_rate,
        betas=(adam_beta1, adam_beta2),
        weight_decay=adam_weight_decay,
        eps=adam_epsilon,
    )

    if is_main_process:
        zero_rank_print(f"trainable params number: {len(trainable_params)}")
        zero_rank_print(f"trainable params scale: {sum(p.numel() for p in trainable_params) / 1e6:.3f} M")

    # Enable gradient checkpointing
    if gradient_checkpointing:
        vae.enable_gradient_checkpointing()

    # Move models to GPU
    vae.to(local_rank)

    # Get the training dataset
    if dataset == 'InternVid':
        train_dataset = InternVidVideo(**train_data)
    elif dataset == 'HAA':
        train_dataset = HAAVideo(**train_data)
    else:
        assert False, "Not Implemented."
    distributed_sampler = DistributedSampler(
        train_dataset,
        num_replicas=num_processes,
        rank=global_rank,
        shuffle=True,
        seed=global_seed,
    )

    # DataLoaders creation:
    train_dataloader = torch.utils.data.DataLoader(
        train_dataset,
        batch_size=train_batch_size,
        shuffle=False,
        sampler=distributed_sampler,
        num_workers=num_workers,
        pin_memory=True,
        drop_last=True,
    )

    # Get the training iteration
    if max_train_steps == -1:
        assert max_train_epoch != -1
        max_train_steps = max_train_epoch * len(train_dataloader)
        
    if checkpointing_steps == -1:
        assert checkpointing_epochs != -1
        checkpointing_steps = checkpointing_epochs * len(train_dataloader)

    if scale_lr:
        learning_rate = (learning_rate * gradient_accumulation_steps * train_batch_size * num_processes)

    # Scheduler
    lr_scheduler = get_scheduler(
        lr_scheduler,
        optimizer=optimizer,
        num_warmup_steps=lr_warmup_steps * gradient_accumulation_steps,
        num_training_steps=max_train_steps * gradient_accumulation_steps,
    )

    # DDP warpper
    vae = DDP(vae, device_ids=[local_rank], output_device=local_rank)

    # We need to recalculate our total training steps as the size of the training dataloader may have changed.
    num_update_steps_per_epoch = math.ceil(len(train_dataloader) / gradient_accumulation_steps)
    # Afterwards we recalculate our number of training epochs
    num_train_epochs = math.ceil(max_train_steps / num_update_steps_per_epoch)

    # Train!
    total_batch_size = train_batch_size * num_processes * gradient_accumulation_steps

    if is_main_process:
        logging.info("***** Running training *****")
        logging.info(f"  Num examples = {len(train_dataset)}")
        logging.info(f"  Num Epochs = {num_train_epochs}")
        logging.info(f"  Instantaneous batch size per device = {train_batch_size}")
        logging.info(f"  Total train batch size (w. parallel, distributed & accumulation) = {total_batch_size}")
        logging.info(f"  Gradient Accumulation steps = {gradient_accumulation_steps}")
        logging.info(f"  Total optimization steps = {max_train_steps}")
    global_step = 0
    first_epoch = 0

    # Only show the progress bar once on each machine.
    progress_bar = tqdm(range(global_step, max_train_steps), disable=not is_main_process)
    progress_bar.set_description("Steps")

    # Support mixed-precision training
    scaler = torch.cuda.amp.GradScaler() if mixed_precision_training else None

    for epoch in range(first_epoch, num_train_epochs):
        train_dataloader.sampler.set_epoch(epoch)
        vae.module.decoder.train()
        
        for step, batch in enumerate(train_dataloader):                
            # Data batch sanity check
            if epoch == first_epoch and step == 0:
                pixel_values = batch['pixel_values'].cpu()
                pixel_values = rearrange(pixel_values, "b f c h w -> b c f h w")
                for idx, pixel_value in enumerate(pixel_values):
                    pixel_value = pixel_value[None, ...]
                    save_videos_grid(pixel_value, f"{output_dir}/sanity_check/{f'{global_rank}-{idx}'}.gif", rescale=True)
                    
            ### >>>> Training >>>> ###
            
            with torch.no_grad():
                if cfg_app_change_video > 0.0:
                    batch['pixel_values'] = batch['pixel_values'].to(local_rank)
                    if cfg_app_change_video_version == "4":
                        batch['pixel_values'], batch['text'] = change_app_video_v4(batch['pixel_values'], batch['text'], cfg_app_change_video_type, cfg_app_change_video)
                # Convert videos to latent space            
                pixel_values = batch["pixel_values"].to(local_rank)
                video_length = pixel_values.shape[1]
                # ### save videos before encoding
                # tmp_sample = (rearrange(pixel_values, "b f c h w -> b c f h w").detach().cpu().float() / 2 + 0.5).clamp(0, 1)
                # save_videos_grid(tmp_sample, "/opt/tiger/original.gif")
                # ###
                ### retrieve pyramid features
                batch_size = pixel_values.shape[0]
                ref_pixel_values = pixel_values[:,0]
                ref_pyramid_features = vae.module.extract_encoder_features(ref_pixel_values)
                ###
                if distort_video_prob > 0:
                    distort_pixel_values = pixel_values.clone()
                    for ii in range(batch_size):
                        random_num = random.random()
                        if random_num <= distort_video_prob:
                            distort_pixel_values[ii] = random_distort_whole_video(random_change_color(random_blur_whole_video(distort_pixel_values[ii])))
                    # ### save videos after distortion
                    # tmp_sample = (rearrange(distort_pixel_values, "b f c h w -> b c f h w").detach().cpu().float() / 2 + 0.5).clamp(0, 1)
                    # save_videos_grid(tmp_sample, "/xxx/xxx/tmp.gif")
                    # assert False
                    # ###
                    distort_pixel_values = rearrange(distort_pixel_values, "b f c h w -> (b f) c h w")
                    pixel_values = rearrange(pixel_values, "b f c h w -> (b f) c h w")
                    latents = vae.module.encode(distort_pixel_values).latent_dist
                    latents = latents.sample() * 0.18125
                else:
                    pixel_values = rearrange(pixel_values, "b f c h w -> (b f) c h w")
                    latents = vae.module.encode(pixel_values).latent_dist
                    latents = latents.sample() * 0.18125
                
            with torch.cuda.amp.autocast(enabled=mixed_precision_training):
                rec_pixel_values = vae.module.decode(latents / 0.18125, ref_pyramid_features).sample
                # ### save videos after encoding-decoding
                # tmp_sample = rearrange(rec_pixel_values.detach().cpu(), "(b f) c h w -> b c f h w", f=video_length)
                # tmp_sample = (tmp_sample.float() / 2 + 0.5).clamp(0, 1)
                # save_videos_grid(tmp_sample, "/opt/tiger/encoder.gif")
                # ###
                loss = F.mse_loss(rec_pixel_values.float(), pixel_values.float(), reduction="mean")
                if spatial_gradient_loss_weight > 0:
                    pixel_values_grad_h = pixel_values[:,:,:-1,:] - pixel_values[:,:,1:,:]
                    pixel_values_grad_w = pixel_values[:,:,:,:-1] - pixel_values[:,:,:,1:]
                    rec_pixel_values_grad_h = rec_pixel_values[:,:,:-1,:] - rec_pixel_values[:,:,1:,:]
                    rec_pixel_values_grad_w = rec_pixel_values[:,:,:,:-1] - rec_pixel_values[:,:,:,1:]
                    spatial_gradient_loss = F.mse_loss(rec_pixel_values_grad_h.float(), pixel_values_grad_h.float(), reduction="mean") + F.mse_loss(rec_pixel_values_grad_w.float(), pixel_values_grad_w.float(), reduction="mean")
                    loss += spatial_gradient_loss_weight * spatial_gradient_loss
            optimizer.zero_grad()

            # Backpropagate
            if mixed_precision_training:
                scaler.scale(loss).backward()
                """ >>> gradient clipping >>> """
                scaler.unscale_(optimizer)
                torch.nn.utils.clip_grad_norm_(trainable_params, max_grad_norm)
                """ <<< gradient clipping <<< """
                scaler.step(optimizer)
                scaler.update()
            else:
                loss.backward()
                """ >>> gradient clipping >>> """
                torch.nn.utils.clip_grad_norm_(trainable_params, max_grad_norm)
                """ <<< gradient clipping <<< """
                optimizer.step()

            lr_scheduler.step()
            progress_bar.update(1)
            global_step += 1
            
            ### <<<< Training <<<< ###
            
            # Wandb logging
            if is_main_process and (not is_debug) and use_wandb:
                wandb.log({"train_loss": loss.item()}, step=global_step)
                
            # Save checkpoint
            if is_main_process and global_step % checkpointing_steps == 0 or global_step in checkpointing_steps_tuple:
                save_path = os.path.join(output_dir, f"checkpoints-vae")
                os.makedirs(save_path, exist_ok=True)
                state_dict = {
                    "epoch": epoch,
                    "global_step": global_step,
                    "state_dict": vae.module.decoder.state_dict(),
                }
                torch.save(state_dict, os.path.join(save_path, f"checkpoint-{global_step}.ckpt"))
                logging.info(f"Saved state to {save_path} (global_step: {global_step})")
                
            # Periodically validation
            if is_main_process and (global_step % validation_steps == 0 or global_step in validation_steps_tuple):
                from decord import VideoReader
                vae.module.decoder.eval()
                
                height = train_data.sample_size[0] if not isinstance(train_data.sample_size, int) else train_data.sample_size
                width  = train_data.sample_size[1] if not isinstance(train_data.sample_size, int) else train_data.sample_size
                        
                controlnet_images_path = validation_data.controlnet_images_path
                image_transforms = transforms.Compose([
                    transforms.Resize((height, width), antialias=True),
                    transforms.ToTensor(),
                ])
                groundtruth_video_path = validation_data.groundtruth_video_path
                video_transforms = transforms.Compose([
                    transforms.Resize((height, width), antialias=True),
                    transforms.Normalize(mean=[0.5, 0.5, 0.5], std=[0.5, 0.5, 0.5], inplace=True),
                ])
                assert len(controlnet_images_path) == len(groundtruth_video_path)
                samples = []

                with torch.no_grad():
                    for idx in range(len(controlnet_images_path)):
                        controlnet_images = image_transforms(Image.open(controlnet_images_path[idx]).convert("RGB")).cuda().unsqueeze(0)
                        ref_pyramid_features = vae.module.encoder.pyramid_feature_forward(controlnet_images * 2. - 1.)
                        
                        video_reader = VideoReader(groundtruth_video_path[idx])
                        video_length = len(video_reader)
                        batch_index = np.linspace(0, video_length - 1, train_data.sample_n_frames, dtype=int)
                        gt_video = torch.from_numpy(video_reader.get_batch(batch_index).asnumpy() / 255.0).permute(0, 3, 1, 2).contiguous().float()
                        ### distort
                        if distort_video_prob > 0:
                            gt_video = distort_whole_video(change_color(blur_whole_video(gt_video)))
                        gt_video = video_transforms(gt_video).cuda()
                        gt_video_latent = vae.module.encode(gt_video).latent_dist.sample() * 0.18215

                        sample = vae.module.decode(gt_video_latent / 0.18125, ref_pyramid_features).sample
                        sample = rearrange(sample.detach().cpu(), "(b f) c h w -> b c f h w", f=train_data.sample_n_frames)
                        sample = (sample.float() / 2 + 0.5).clamp(0, 1)
                        save_videos_grid(sample, f"{output_dir}/samples/sample-{global_step}/{idx}.gif")
                        samples.append(sample)
                
                samples = torch.concat(samples)
                save_path = f"{output_dir}/samples/sample-{global_step}.gif"
                save_videos_grid(samples, save_path)

                logging.info(f"Saved samples to {save_path}")

                vae.module.decoder.train()
                
            logs = {"step_loss": loss.detach().item(), "lr": lr_scheduler.get_last_lr()[0]}
            progress_bar.set_postfix(**logs)
            
            if global_step >= max_train_steps:
                break

            torch.cuda.empty_cache()
            
    dist.destroy_process_group()



if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--config",   type=str, required=True)
    parser.add_argument("--launcher", type=str, choices=["pytorch", "slurm"], default="pytorch")
    parser.add_argument("--wandb",    action="store_true")
    args = parser.parse_args()

    # name   = Path(args.config).stem
    name = args.config.split('/')[-3] + '_' + args.config.split('/')[-2] + '_' + args.config.split('/')[-1].split('.')[0]
    config = OmegaConf.load(args.config)

    main(name=name, launcher=args.launcher, use_wandb=args.wandb, **config)
