import argparse
from concurrent import futures
from pathlib import Path

import cv2
from tqdm.auto import tqdm


def extract_middle_frame(video_path, output_dir):
    video_id = Path(video_path).stem
    cap = cv2.VideoCapture(str(video_path))

    frame_count = int(cap.get(cv2.CAP_PROP_FRAME_COUNT))

    middle_frame_idx = frame_count // 2  # Calculate the index of the middle frame

    cap.set(cv2.CAP_PROP_POS_FRAMES, middle_frame_idx)
    ret, frame = cap.read()

    if ret:
        frame_id = str(middle_frame_idx).zfill(3)
        output_path = str(Path(output_dir) / f"{video_id}_{frame_id}.png")

        cv2.imwrite(output_path, frame)

    cap.release()


def read_video_list(video_list_file):
    """
    Reads the video list file and returns a list of video paths.
    """
    video_paths = []
    with open(video_list_file, "r") as f:
        for line in f:
            video_path = line.strip()
            video_paths.append(video_path)
    return video_paths


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Extract middle frames from videos.")
    parser.add_argument("root_dir", help="path to the root directory")
    parser.add_argument(
        "subdirectory", type=int, help="subdirectory to extract videos from"
    )
    parser.add_argument("output_dir", help="path to the output directory")
    parser.add_argument("video_list_file", help="path to the video list file")
    args = parser.parse_args()

    video_paths = read_video_list(args.video_list_file)
    shard_video_paths = [
        pth for pth in video_paths if Path(pth).parent.name == str(args.subdirectory)
    ]
    output_dir = Path(args.output_dir) / str(args.subdirectory)
    output_dir.mkdir(exist_ok=True)

    for video_path in tqdm(shard_video_paths):
        with futures.ThreadPoolExecutor(max_workers=4) as executor:
            for video_filename in shard_video_paths:
                full_video_path = Path(args.root_dir) / video_filename
                if full_video_path.exists():
                    executor.submit(extract_middle_frame, full_video_path, output_dir)

            # kill the executor
            executor.shutdown(wait=True)
            break
