# Complete Optimization Problem and Solution: company_office

## 1. Problem Context and Goals

### Context  
The company is focused on optimizing the allocation of office spaces across multiple buildings to minimize the total leasing cost. The decision involves determining the amount of space to allocate to each company in each building, ensuring that each company's space requirements are met based on their sales and assets. The cost per square foot for leasing space varies by building, and the total available space in each building is limited. The goal is to allocate space in a way that minimizes the overall leasing cost while adhering to these constraints.

Key operational parameters include:
- **Cost per square foot for leasing space in each building**: This is used to calculate the total leasing cost, which is the primary metric to minimize.
- **Minimum space required by each company**: This ensures that each company receives enough space to meet its operational needs based on its sales and assets.
- **Total available space in each building**: This ensures that the total space allocated in each building does not exceed its capacity.

The total leasing cost is calculated by summing the product of the cost per square foot for each building and the space allocated to each company in that building. This calculation is linear and directly tied to the decision variables, ensuring a straightforward optimization formulation.

### Goals  
The primary goal of this optimization problem is to minimize the total leasing cost of office spaces across all buildings. This is achieved by determining the optimal allocation of space to each company in each building, considering the cost per square foot for each building and the space requirements of each company. Success is measured by achieving the lowest possible total leasing cost while ensuring that all companies receive their required space and that no building exceeds its available space capacity.

## 2. Constraints    

The optimization problem is subject to the following constraints:
1. **Minimum Space Requirement**: The total space allocated to each company across all buildings must meet or exceed the company's minimum space requirement. This ensures that each company has enough space to operate effectively based on its sales and assets.
2. **Building Capacity**: The total space allocated in each building must not exceed the building's available space. This ensures that the physical limitations of each building are respected.
3. **Non-Negative Allocation**: The space allocated to each company in each building must be non-negative. This ensures that the allocation is physically meaningful and feasible.

These constraints are linear and directly tied to the decision variables, ensuring that the optimization problem remains linear and tractable.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 2 Database Schema
-- Objective: Added space_allocated table to map decision variables, updated business configuration logic with scalar parameters and formulas, and ensured schema follows normalization principles.

CREATE TABLE cost_per_sqft (
  building_id INTEGER,
  cost_per_sqft FLOAT
);

CREATE TABLE required_space (
  company_id INTEGER,
  required_space INTEGER
);

CREATE TABLE available_space (
  building_id INTEGER,
  available_space INTEGER
);

CREATE TABLE space_allocated (
  building_id INTEGER,
  company_id INTEGER,
  space_allocated FLOAT
);
```

### Data Dictionary  
- **cost_per_sqft**: This table contains the cost per square foot for leasing space in each building. It is used to calculate the total leasing cost in the objective function.
  - **building_id**: Unique identifier for each building.
  - **cost_per_sqft**: Cost per square foot for leasing space in the building.
  
- **required_space**: This table contains the minimum space required by each company based on their sales and assets. It is used to ensure that each company's space requirements are met in the constraints.
  - **company_id**: Unique identifier for each company.
  - **required_space**: Minimum space required by the company.
  
- **available_space**: This table contains the total available space in each building. It is used to ensure that the total space allocated in each building does not exceed its capacity in the constraints.
  - **building_id**: Unique identifier for each building.
  - **available_space**: Total available space in the building.
  
- **space_allocated**: This table contains the amount of space allocated to each company in each building. It represents the decision variables in the optimization problem.
  - **building_id**: Unique identifier for each building.
  - **company_id**: Unique identifier for each company.
  - **space_allocated**: Amount of space allocated to the company in the building.


### Retrieved Values

**Query 1: Query Description: Retrieve the cost per square foot for each building. This is crucial for calculating the total leasing cost, which is the primary metric to minimize in the optimization problem.**

```sql
SELECT building_id, cost_per_sqft FROM cost_per_sqft;
```

**Results (CSV format):**
```csv
building_id,cost_per_sqft
1,55.0
2,65.0
3,45.0
```

**Query 2: Query Description: Retrieve the minimum space required by each company. This data is essential for ensuring that each company's space requirements are met in the constraints.**

```sql
SELECT company_id, required_space FROM required_space;
```

**Results (CSV format):**
```csv
company_id,required_space
1,1200
2,1800
3,900
```

**Query 3: Query Description: Retrieve the total available space in each building. This information is necessary to ensure that the total space allocated in each building does not exceed its capacity.**

```sql
SELECT building_id, available_space FROM available_space;
```

**Results (CSV format):**
```csv
building_id,available_space
1,6000
2,7000
3,5000
```

**Query 4: Query Description: Retrieve the current space allocated to each company in each building. This represents the decision variables in the optimization problem and is needed to evaluate the current allocation and make adjustments.**

```sql
SELECT building_id, company_id, space_allocated FROM space_allocated;
```

**Results (CSV format):**
```csv
building_id,company_id,space_allocated
1,1,600.0
1,2,800.0
2,3,900.0
```

**Query 5: Query Description: Calculate the total space allocated in each building to ensure it does not exceed the available space. This aggregated data is important for the building capacity constraint.**

```sql
SELECT building_id, SUM(space_allocated) AS total_allocated_space FROM space_allocated GROUP BY building_id;
```

**Results (CSV format):**
```csv
building_id,total_allocated_space
1,1400.0
2,900.0
```

**Query 6: Query Description: Calculate the total space allocated to each company across all buildings to ensure it meets or exceeds the company's minimum space requirement. This aggregated data is important for the minimum space requirement constraint.**

```sql
SELECT company_id, SUM(space_allocated) AS total_allocated_space FROM space_allocated GROUP BY company_id;
```

**Results (CSV format):**
```csv
company_id,total_allocated_space
1,600.0
2,800.0
3,900.0
```

**Query 7: Query Description: Join the space_allocated table with the cost_per_sqft table to calculate the total leasing cost for each company in each building. This data is essential for the objective function to minimize the total leasing cost.**

```sql
SELECT sa.building_id, sa.company_id, sa.space_allocated, cps.cost_per_sqft, sa.space_allocated * cps.cost_per_sqft AS leasing_cost FROM space_allocated sa JOIN cost_per_sqft cps ON sa.building_id = cps.building_id;
```

**Results (CSV format):**
```csv
building_id,company_id,space_allocated,cost_per_sqft,leasing_cost
1,1,600.0,55.0,33000.0
1,2,800.0,55.0,44000.0
2,3,900.0,65.0,58500.0
```

**Query 8: Query Description: Join the space_allocated table with the required_space table to compare the allocated space with the required space for each company. This helps in verifying the minimum space requirement constraint.**

```sql
SELECT sa.company_id, rs.required_space, SUM(sa.space_allocated) AS total_allocated_space FROM space_allocated sa JOIN required_space rs ON sa.company_id = rs.company_id GROUP BY sa.company_id, rs.required_space;
```

**Results (CSV format):**
```csv
company_id,required_space,total_allocated_space
1,1200,600.0
2,1800,800.0
3,900,900.0
```

**Query 9: Query Description: Join the space_allocated table with the available_space table to compare the allocated space with the available space in each building. This helps in verifying the building capacity constraint.**

```sql
SELECT sa.building_id, avs.available_space, SUM(sa.space_allocated) AS total_allocated_space FROM space_allocated sa JOIN available_space avs ON sa.building_id = avs.building_id GROUP BY sa.building_id, avs.available_space;
```

**Results (CSV format):**
```csv
building_id,available_space,total_allocated_space
1,6000,1400.0
2,7000,900.0
```

**Query 10: Query Description: Retrieve a summary of the total leasing cost across all buildings and companies. This aggregated data is useful for evaluating the overall cost and the effectiveness of the optimization.**

```sql
SELECT SUM(sa.space_allocated * cps.cost_per_sqft) AS total_leasing_cost FROM space_allocated sa JOIN cost_per_sqft cps ON sa.building_id = cps.building_id;
```

**Results (CSV format):**
```csv
total_leasing_cost
135500.0
```

