# Complete Optimization Problem and Solution: device

## 1. Problem Context and Goals

### Context  
A retail chain is focused on optimizing the distribution of devices across its shops to minimize shipping costs while ensuring each shop meets its demand and does not exceed its storage capacity. The key decision involves determining the number of each device type to be shipped from a central warehouse to each shop. This decision must account for the shipping cost associated with each device and shop combination, the demand for each device at each shop, and the maximum storage capacity of each shop.  

The shipping cost for each device to each shop is a fixed value, and the total shipping cost is calculated by multiplying the shipping cost by the number of devices shipped. The business configuration includes a maximum storage capacity for each shop, which ensures that the total number of devices shipped to a shop does not exceed its storage limit. This setup naturally leads to a linear optimization formulation, as all relationships are proportional and additive, without any nonlinear interactions such as variable products or divisions.  

### Goals  
The primary goal of this optimization problem is to minimize the total shipping cost incurred by the retail chain. This is achieved by determining the optimal number of devices to ship to each shop, ensuring that the sum of shipping costs across all device and shop combinations is as low as possible. Success is measured by the ability to meet all shop demands while respecting storage capacity limits and ensuring that all shipments are non-negative. The optimization aligns with the business objective of cost efficiency in device distribution.  

## 2. Constraints  

The optimization problem is subject to the following constraints:  
1. **Demand Fulfillment**: The total number of devices shipped to each shop must meet or exceed the demand for each device at that shop. This ensures that all shops have sufficient inventory to meet customer needs.  
2. **Storage Capacity**: The total number of devices shipped to each shop must not exceed the shop’s maximum storage capacity. This ensures that shops are not overstocked, which could lead to operational inefficiencies.  
3. **Non-Negative Shipments**: The number of devices shipped to each shop must be a non-negative integer. This ensures that the solution is practical and feasible in a real-world business context.  

These constraints are expressed in linear terms, ensuring that the optimization problem remains straightforward and computationally tractable.  

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 1 Database Schema
-- Objective: Schema changes include creating tables for shipping costs, demand, and storage capacity. Business configuration logic updated with scalar parameters and formulas.

CREATE TABLE shipping_cost (
  device_id INTEGER,
  shop_id INTEGER,
  cost FLOAT
);

CREATE TABLE demand (
  device_id INTEGER,
  shop_id INTEGER,
  quantity INTEGER
);

CREATE TABLE storage_capacity (
  shop_id INTEGER,
  capacity INTEGER
);

CREATE TABLE stock (
  device_id INTEGER,
  shop_id INTEGER,
  quantity INTEGER
);
```

### Data Dictionary  
- **Shipping Cost**:  
  - **Purpose**: Represents the cost to ship a specific device to a specific shop.  
  - **Optimization Role**: Used as coefficients in the objective function to calculate total shipping costs.  
  - **Columns**:  
    - `device_id`: Identifier for the device.  
    - `shop_id`: Identifier for the shop.  
    - `cost`: Shipping cost for the device to the shop.  

- **Demand**:  
  - **Purpose**: Represents the demand for a specific device at a specific shop.  
  - **Optimization Role**: Used as bounds in the demand fulfillment constraint.  
  - **Columns**:  
    - `device_id`: Identifier for the device.  
    - `shop_id`: Identifier for the shop.  
    - `quantity`: Demand quantity for the device at the shop.  

- **Storage Capacity**:  
  - **Purpose**: Represents the maximum number of devices a shop can store.  
  - **Optimization Role**: Used as bounds in the storage capacity constraint.  
  - **Columns**:  
    - `shop_id`: Identifier for the shop.  
    - `capacity`: Maximum storage capacity for the shop.  

- **Stock**:  
  - **Purpose**: Represents the number of devices to be shipped to a specific shop.  
  - **Optimization Role**: Used as decision variables in the optimization model.  
  - **Columns**:  
    - `device_id`: Identifier for the device.  
    - `shop_id`: Identifier for the shop.  
    - `quantity`: Number of devices to be shipped.  


### Retrieved Values

**Query 1: Query Description: Retrieve the shipping cost for each device to each shop. This is crucial for the objective function to calculate the total shipping cost.**

```sql
SELECT device_id, shop_id, cost FROM shipping_cost;
```

**Results (CSV format):**
```csv
device_id,shop_id,cost
1,101,12.5
2,102,18.0
3,103,15.0
```

**Query 2: Query Description: Retrieve the demand for each device at each shop. This is essential for the demand fulfillment constraint to ensure that each shop's demand is met.**

```sql
SELECT device_id, shop_id, quantity FROM demand;
```

**Results (CSV format):**
```csv
device_id,shop_id,quantity
1,101,60
2,102,80
3,103,70
```

**Query 3: Query Description: Retrieve the storage capacity for each shop. This is necessary for the storage capacity constraint to ensure that the total number of devices shipped does not exceed the shop's capacity.**

```sql
SELECT shop_id, capacity FROM storage_capacity;
```

**Results (CSV format):**
```csv
shop_id,capacity
101,250
102,300
103,200
```

**Query 4: Query Description: Retrieve the current stock of each device at each shop. This is important to understand the initial inventory and to calculate the additional devices needed to meet demand.**

```sql
SELECT device_id, shop_id, quantity FROM stock;
```

**Results (CSV format):**
```csv
device_id,shop_id,quantity
1,101,20
2,102,30
3,103,25
```

**Query 5: Query Description: Retrieve the total demand for each shop across all devices. This helps in understanding the overall demand per shop and can be used to cross-check against storage capacity.**

```sql
SELECT shop_id, SUM(quantity) AS total_demand FROM demand GROUP BY shop_id;
```

**Results (CSV format):**
```csv
shop_id,total_demand
101,60
102,80
103,70
```

**Query 6: Query Description: Retrieve the total shipping cost per shop across all devices. This provides a summary of shipping costs per shop, which can be useful for cost analysis and optimization.**

```sql
SELECT shop_id, SUM(cost) AS total_shipping_cost FROM shipping_cost GROUP BY shop_id;
```

**Results (CSV format):**
```csv
shop_id,total_shipping_cost
101,12.5
102,18.0
103,15.0
```

**Query 7: Query Description: Retrieve the total storage capacity and total demand for each shop. This helps in identifying shops where demand exceeds capacity, which is critical for constraint validation.**

```sql
SELECT sc.shop_id, sc.capacity AS total_capacity, SUM(d.quantity) AS total_demand FROM storage_capacity sc JOIN demand d ON sc.shop_id = d.shop_id GROUP BY sc.shop_id, sc.capacity;
```

**Results (CSV format):**
```csv
shop_id,total_capacity,total_demand
101,250,60
102,300,80
103,200,70
```

**Query 8: Query Description: Retrieve the shipping cost, demand, and storage capacity for each device and shop combination. This comprehensive view is useful for detailed analysis and optimization.**

```sql
SELECT sc.device_id, sc.shop_id, sc.cost, d.quantity AS demand, sc2.capacity FROM shipping_cost sc JOIN demand d ON sc.device_id = d.device_id AND sc.shop_id = d.shop_id JOIN storage_capacity sc2 ON sc.shop_id = sc2.shop_id;
```

**Results (CSV format):**
```csv
device_id,shop_id,cost,demand,capacity
1,101,12.5,60,250
2,102,18.0,80,300
3,103,15.0,70,200
```

**Query 9: Query Description: Retrieve the current stock and demand for each device at each shop. This helps in identifying the gap between current stock and demand, which is essential for determining the number of devices to ship.**

```sql
SELECT s.device_id, s.shop_id, s.quantity AS current_stock, d.quantity AS demand FROM stock s JOIN demand d ON s.device_id = d.device_id AND s.shop_id = d.shop_id;
```

**Results (CSV format):**
```csv
device_id,shop_id,current_stock,demand
1,101,20,60
2,102,30,80
3,103,25,70
```

**Query 10: Query Description: Retrieve the total number of devices that can be shipped to each shop without exceeding its storage capacity. This is useful for ensuring that the storage capacity constraint is respected.**

```sql
SELECT sc.shop_id, sc.capacity - IFNULL(SUM(s.quantity), 0) AS remaining_capacity FROM storage_capacity sc LEFT JOIN stock s ON sc.shop_id = s.shop_id GROUP BY sc.shop_id, sc.capacity;
```

**Results (CSV format):**
```csv
shop_id,remaining_capacity
101,230
102,270
103,175
```

