# Complete Optimization Problem and Solution: insurance_policies

## 1. Problem Context and Goals

### Context  
The insurance company is focused on minimizing the total cost of settling claims while ensuring fairness and adherence to budget constraints. The company must allocate settlement amounts across multiple claims in a way that reduces the overall payout. This allocation is subject to several operational constraints:  
- The total amount settled across all claims must not exceed the total budget available for settlements.  
- The amount settled for each individual claim must not exceed the maximum settlement amount allowed per claim.  
- The amount settled for each claim must be at least a minimum percentage of the claimed amount, ensuring fairness to policyholders.  

The total budget available for settlements is currently set at $500,000, and the maximum amount that can be settled per claim is $50,000. Additionally, the company ensures that each settlement is at least 50% of the claimed amount. These parameters are critical for maintaining financial stability and fairness in the settlement process.  

### Goals  
The primary goal of this optimization problem is to minimize the total amount settled across all claims. This is achieved by determining the optimal settlement amount for each claim while adhering to the constraints mentioned above. Success is measured by achieving the lowest possible total settlement cost without violating any of the operational or fairness constraints.  

## 2. Constraints  

The optimization problem is subject to the following constraints, which ensure that the settlement process remains within operational and financial limits:  
1. **Total Budget Constraint**: The sum of all settlement amounts across claims must not exceed the total budget available for settlements, which is $500,000.  
2. **Maximum Settlement per Claim**: The settlement amount for any individual claim must not exceed the maximum allowed settlement per claim, which is $50,000.  
3. **Minimum Settlement Percentage**: The settlement amount for each claim must be at least 50% of the claimed amount, ensuring a fair minimum payout to policyholders.  

These constraints are designed to align with the company’s financial and operational policies while maintaining fairness in the settlement process.  

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 1 Database Schema
-- Objective: Schema changes include adding a Budgets table for Total_Budget and Maximum_Settlement_Per_Claim. Configuration logic updated with scalar parameters for Total_Budget and Maximum_Settlement_Per_Claim, and a business logic formula for minimum settlement percentage.

CREATE TABLE Claims (
  Claim_ID INTEGER,
  Amount_Claimed FLOAT
);

CREATE TABLE Settlements (
  Claim_ID INTEGER,
  Amount_Settled FLOAT
);

CREATE TABLE Budgets (
  Total_Budget FLOAT,
  Maximum_Settlement_Per_Claim FLOAT
);
```

### Data Dictionary  
- **Claims Table**:  
  - **Claim_ID**: A unique identifier for each insurance claim.  
  - **Amount_Claimed**: The amount requested by the policyholder for each claim. This value is used to determine the minimum settlement amount per claim.  

- **Settlements Table**:  
  - **Claim_ID**: A unique identifier for each claim, linking it to the corresponding settlement.  
  - **Amount_Settled**: The amount allocated to settle each claim. This is the decision variable to be optimized.  

- **Budgets Table**:  
  - **Total_Budget**: The total amount of money available for settling all claims. This value is used as an upper bound for the sum of all settlements.  
  - **Maximum_Settlement_Per_Claim**: The maximum amount that can be allocated to settle any single claim. This value is used as an upper bound for individual settlements.  


### Retrieved Values

**Query 1: This is crucial for setting the upper bounds in the optimization constraints.**

```sql
SELECT Total_Budget, Maximum_Settlement_Per_Claim FROM Budgets;
```

**Results (CSV format):**
```csv
Total_Budget,Maximum_Settlement_Per_Claim
500000.0,50000.0
```

**Query 2: This data is essential for calculating the minimum settlement amount per claim (50% of the claimed amount).**

```sql
SELECT Claim_ID, Amount_Claimed FROM Claims;
```

**Results (CSV format):**
```csv
Claim_ID,Amount_Claimed
1,15000.0
2,25000.0
3,35000.0
```

**Query 3: This data is necessary to understand the current state of settlements and to ensure that the optimization respects existing settlements.**

```sql
SELECT Claim_ID, Amount_Settled FROM Settlements;
```

**Results (CSV format):**
```csv
Claim_ID,Amount_Settled
1,7500.0
2,12500.0
3,17500.0
```

**Query 4: This helps in identifying claims that need to be considered for new settlements.**

```sql
SELECT c.Claim_ID, c.Amount_Claimed FROM Claims c LEFT JOIN Settlements s ON c.Claim_ID = s.Claim_ID WHERE s.Amount_Settled IS NULL;
```

**Results (CSV format):**
```csv
Claim_ID,Amount_Claimed
```

**Query 5: This is important to ensure that the sum of new settlements does not exceed the remaining budget.**

```sql
SELECT SUM(Amount_Settled) AS Total_Settled_So_Far FROM Settlements;
```

**Results (CSV format):**
```csv
Total_Settled_So_Far
37500.0
```

**Query 6: This is critical for ensuring that each settlement meets the minimum fairness requirement.**

```sql
SELECT Claim_ID, Amount_Claimed, Amount_Claimed * 0.5 AS Minimum_Settlement FROM Claims;
```

**Results (CSV format):**
```csv
Claim_ID,Amount_Claimed,Minimum_Settlement
1,15000.0,7500.0
2,25000.0,12500.0
3,35000.0,17500.0
```

**Query 7: This helps in understanding the range within which each settlement can be optimized.**

```sql
SELECT c.Claim_ID, c.Amount_Claimed, c.Amount_Claimed * 0.5 AS Minimum_Settlement, b.Maximum_Settlement_Per_Claim FROM Claims c CROSS JOIN Budgets b;
```

**Results (CSV format):**
```csv
Claim_ID,Amount_Claimed,Minimum_Settlement,Maximum_Settlement_Per_Claim
1,15000.0,7500.0,50000.0
2,25000.0,12500.0,50000.0
3,35000.0,17500.0,50000.0
```

**Query 8: This is essential for ensuring that the sum of new settlements does not exceed the remaining budget.**

```sql
SELECT b.Total_Budget - COALESCE(SUM(s.Amount_Settled), 0) AS Remaining_Budget FROM Budgets b LEFT JOIN Settlements s ON 1=1;
```

**Results (CSV format):**
```csv
Remaining_Budget
462500.0
```

**Query 9: This provides a comprehensive view of the constraints and available resources for optimization.**

```sql
SELECT c.Claim_ID, c.Amount_Claimed, c.Amount_Claimed * 0.5 AS Minimum_Settlement, b.Total_Budget - COALESCE(SUM(s.Amount_Settled), 0) AS Remaining_Budget FROM Claims c CROSS JOIN Budgets b LEFT JOIN Settlements s ON 1=1 GROUP BY c.Claim_ID, c.Amount_Claimed, b.Total_Budget;
```

**Results (CSV format):**
```csv
Claim_ID,Amount_Claimed,Minimum_Settlement,Remaining_Budget
1,15000.0,7500.0,462500.0
2,25000.0,12500.0,462500.0
3,35000.0,17500.0,462500.0
```

