# Complete PYOMO implementation - Retry Attempt 1

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def climbing_optimization():
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    climber_ids = [1, 2, 3]
    mountain_ids = [1, 2, 3]
    
    points = {
        (1, 1): 10,
        (2, 2): 20,
        (3, 3): 30
    }
    
    time_spent = {
        (1, 1): 5,
        (2, 2): 10,
        (3, 3): 15
    }
    
    total_time_limit = 100
    
    # 3. SETS
    model.C = pyo.Set(initialize=climber_ids)
    model.M = pyo.Set(initialize=mountain_ids)
    
    # 4. PARAMETERS
    model.points = pyo.Param(model.C, model.M, initialize=points, default=0)
    model.time = pyo.Param(model.C, model.M, initialize=time_spent, default=0)
    model.total_time_limit = pyo.Param(initialize=total_time_limit)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.C, model.M, within=pyo.Binary)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.points[i, j] * model.x[i, j] for i in model.C for j in model.M)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    def total_time_constraint_rule(model):
        return sum(model.time[i, j] * model.x[i, j] for i in model.C for j in model.M) <= model.total_time_limit
    model.total_time_constraint = pyo.Constraint(rule=total_time_constraint_rule)
    
    def single_assignment_constraint_rule(model, i):
        return sum(model.x[i, j] for j in model.M) == 1
    model.single_assignment_constraint = pyo.Constraint(model.C, rule=single_assignment_constraint_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)  # tee=True shows solver output
    
    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print(f"Optimal value: {pyo.value(model.objective)}")
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

if __name__ == "__main__":
    climbing_optimization()