# Complete Optimization Problem and Solution: coffee_shop

## 1. Problem Context and Goals

### Context  
A coffee shop chain is focused on optimizing staff allocation during happy hours across its various shops. The goal is to maximize customer satisfaction by ensuring that members, who are valued based on their membership level, spend more during these peak hours. The chain must allocate staff efficiently to meet customer demand while staying within operational constraints. 

The key decision involves determining the number of staff to assign to each shop during happy hours. This decision must consider the total budget for staff across all shops, which is capped at 150. Additionally, each shop has specific staffing limits: a minimum of 2 staff members is required to maintain basic operations, and a maximum of 15 staff members is allowed to handle peak customer traffic. The chain also enforces a maximum spending limit of $50 per member to control costs while encouraging customer spending.

The optimization problem is designed to maximize the total amount spent by members during happy hours, weighted by their membership level. This ensures that higher-tier members, who contribute more to revenue, are prioritized. The problem is structured to ensure that all constraints are linear, avoiding any nonlinear relationships such as variable products or divisions.

### Goals  
The primary goal of this optimization is to maximize the total weighted spending by members during happy hours. This is achieved by assigning the optimal number of staff to each shop, ensuring that higher-tier members are adequately served. Success is measured by the total amount spent by members, adjusted for their membership level, while adhering to the total staff budget and individual shop staffing limits. The optimization ensures that the chain operates efficiently, balancing customer satisfaction with operational costs.

## 2. Constraints  

The optimization problem is subject to the following constraints:  
1. **Total Staff Budget Constraint**: The total number of staff assigned across all shops must not exceed the total budget of 150. This ensures that the chain operates within its financial limits.  
2. **Shop Staffing Limits**: Each shop must have a minimum of 2 staff members to maintain basic operations and a maximum of 15 staff members to handle peak customer traffic. This ensures that each shop is adequately staffed without exceeding its capacity.  
3. **Maximum Spending Limit per Member**: The spending per member during happy hours must not exceed $50. This constraint helps control costs while encouraging customer spending.  

These constraints are designed to ensure that the optimization problem remains linear, avoiding any nonlinear relationships that could complicate the solution.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 1 Database Schema
-- Objective: Schema changes include creating new tables for missing constraint bounds, modifying existing tables to refine decision variable mapping, and adding business configuration logic for scalar parameters and formulas.

CREATE TABLE happy_hour_member (
  Total_amount FLOAT
);

CREATE TABLE member (
  Level_of_membership INTEGER
);

CREATE TABLE happy_hour (
  Num_of_staff_in_charge INTEGER
);

CREATE TABLE shop_staff_limits (
  Max_Staff_Per_Shop INTEGER,
  Min_Staff_Per_Shop INTEGER
);
```

### Data Dictionary  
- **happy_hour_member**:  
  - **Business Purpose**: Records the amount spent by each member during happy hours.  
  - **Optimization Role**: Provides coefficients for the objective function.  
  - **Columns**:  
    - **Total_amount**: The amount spent by a member during a happy hour.  

- **member**:  
  - **Business Purpose**: Stores membership information for each member.  
  - **Optimization Role**: Provides coefficients for the objective function.  
  - **Columns**:  
    - **Level_of_membership**: The membership level of a member, used to weight their spending in the objective function.  

- **happy_hour**:  
  - **Business Purpose**: Records the number of staff assigned to each shop during happy hours.  
  - **Optimization Role**: Represents the decision variable in the optimization model.  
  - **Columns**:  
    - **Num_of_staff_in_charge**: The number of staff assigned to a shop during a happy hour.  

- **shop_staff_limits**:  
  - **Business Purpose**: Stores the maximum and minimum number of staff allowed per shop.  
  - **Optimization Role**: Provides constraint bounds for the optimization model.  
  - **Columns**:  
    - **Max_Staff_Per_Shop**: The maximum number of staff allowed per shop.  
    - **Min_Staff_Per_Shop**: The minimum number of staff required per shop.  

### Current Stored Values  
```sql
-- Iteration 1 Realistic Data
-- Generated by triple expert (business + data + optimization)
-- Values were determined based on typical coffee shop operations, considering factors like average spending, staff requirements, and membership levels. Data was generated to ensure a balance between customer satisfaction and operational costs.

-- Realistic data for happy_hour_member
INSERT INTO happy_hour_member (Total_amount) VALUES (25.5);
INSERT INTO happy_hour_member (Total_amount) VALUES (30.0);
INSERT INTO happy_hour_member (Total_amount) VALUES (15.75);

-- Realistic data for member
INSERT INTO member (Level_of_membership) VALUES (1);
INSERT INTO member (Level_of_membership) VALUES (2);
INSERT INTO member (Level_of_membership) VALUES (3);

-- Realistic data for happy_hour
INSERT INTO happy_hour (Num_of_staff_in_charge) VALUES (3);
INSERT INTO happy_hour (Num_of_staff_in_charge) VALUES (5);
INSERT INTO happy_hour (Num_of_staff_in_charge) VALUES (7);

-- Realistic data for shop_staff_limits
INSERT INTO shop_staff_limits (Max_Staff_Per_Shop, Min_Staff_Per_Shop) VALUES (10, 2);
INSERT INTO shop_staff_limits (Max_Staff_Per_Shop, Min_Staff_Per_Shop) VALUES (12, 3);
INSERT INTO shop_staff_limits (Max_Staff_Per_Shop, Min_Staff_Per_Shop) VALUES (15, 4);
```

## 4. Mathematical Optimization Formulation

#### Decision Variables
- Let \( x_i \) be the number of staff assigned to shop \( i \) during happy hours.  
  - **Type**: Continuous or Integer (depending on the problem's requirements).  
  - **Range**: \( x_i \geq 0 \).  

#### Objective Function
Maximize the total weighted spending by members during happy hours:  
\[
\text{Maximize } Z = \sum_{i} w_i \cdot s_i \cdot x_i
\]  
Where:  
- \( w_i \) is the weight based on the membership level of members in shop \( i \).  
- \( s_i \) is the average spending per member in shop \( i \).  

**Coefficients**:  
- \( w_i \): `member.Level_of_membership` (weight based on membership level).  
- \( s_i \): `happy_hour_member.Total_amount` (average spending per member).  

#### Constraints
1. **Total Staff Budget Constraint**:  
   \[
   \sum_{i} x_i \leq 150
   \]  
   - **Coefficient**: 1 (each staff member counts equally toward the budget).  

2. **Shop Staffing Limits**:  
   \[
   \text{Min_Staff_Per_Shop}_i \leq x_i \leq \text{Max_Staff_Per_Shop}_i \quad \forall i
   \]  
   - **Coefficients**:  
     - Lower bound: `shop_staff_limits.Min_Staff_Per_Shop`.  
     - Upper bound: `shop_staff_limits.Max_Staff_Per_Shop`.  

3. **Maximum Spending Limit per Member**:  
   \[
   s_i \leq 50 \quad \forall i
   \]  
   - **Coefficient**: 1 (spending per member must not exceed $50).  

#### Data Source Verification
- **Objective Function Coefficients**:  
  - \( w_i \): `member.Level_of_membership`.  
  - \( s_i \): `happy_hour_member.Total_amount`.  
- **Constraint Coefficients**:  
  - Total Staff Budget: Constant value of 150.  
  - Shop Staffing Limits: `shop_staff_limits.Min_Staff_Per_Shop` and `shop_staff_limits.Max_Staff_Per_Shop`.  
  - Maximum Spending Limit: Constant value of 50.  

This formulation ensures a linear optimization model that maximizes weighted spending while adhering to the specified constraints.

## 5. Gurobipy Implementation

```python
#!/usr/bin/env python3
"""
Gurobipy 12.0.2 Implementation for Coffee Shop Staff Allocation Optimization
"""

import gurobipy as gp
from gurobipy import GRB

def coffee_shop_optimization():
    """Optimize staff allocation for coffee shops during happy hours."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("coffee_shop_staff_allocation")
    
    # Data from the database
    total_amounts = [25.5, 30.0, 15.75]  # happy_hour_member.Total_amount
    membership_levels = [1, 2, 3]        # member.Level_of_membership
    max_staff_per_shop = [10, 12, 15]    # shop_staff_limits.Max_Staff_Per_Shop
    min_staff_per_shop = [2, 3, 4]       # shop_staff_limits.Min_Staff_Per_Shop
    
    # Number of shops
    n_shops = len(total_amounts)
    
    # CRITICAL: Validate array lengths before loops
    assert len(total_amounts) == len(membership_levels) == len(max_staff_per_shop) == len(min_staff_per_shop) == n_shops, "Array length mismatch"
    
    # 2. VARIABLES
    # Decision variable: Number of staff assigned to each shop
    x = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"x_{i}", lb=min_staff_per_shop[i], ub=max_staff_per_shop[i]) 
         for i in range(n_shops)}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total weighted spending by members during happy hours
    model.setObjective(
        gp.quicksum(membership_levels[i] * total_amounts[i] * x[i] for i in range(n_shops)), 
        GRB.MAXIMIZE
    )
    
    # 4. CONSTRAINTS
    
    # Total Staff Budget Constraint
    model.addConstr(
        gp.quicksum(x[i] for i in range(n_shops)) <= 150, 
        name="total_staff_budget"
    )
    
    # Maximum Spending Limit per Member
    for i in range(n_shops):
        model.addConstr(
            total_amounts[i] <= 50, 
            name=f"max_spending_{i}"
        )
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_shops):
            print(f"Staff assigned to shop {i}: {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
if __name__ == "__main__":
    coffee_shop_optimization()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 1683.75
**Execution Time**: 0.18 seconds
**Reliability**: high
**Analysis**: Gurobipy successfully solved the problem and returned an optimal solution with a value of 1683.75. The execution time was very fast (0.178 seconds), indicating high efficiency.

## 6. DOCplex Implementation

```python
# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_coffee_shop_staff_allocation():
    """Optimize staff allocation across coffee shops during happy hours."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="coffee_shop_staff_allocation")
    
    # Data from the database
    total_amounts = [25.5, 30.0, 15.75]  # happy_hour_member.Total_amount
    membership_levels = [1, 2, 3]        # member.Level_of_membership
    max_staff_per_shop = [10, 12, 15]    # shop_staff_limits.Max_Staff_Per_Shop
    min_staff_per_shop = [2, 3, 4]       # shop_staff_limits.Min_Staff_Per_Shop
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(total_amounts) == len(membership_levels) == len(max_staff_per_shop) == len(min_staff_per_shop), "Array length mismatch"
    safe_range = range(min(len(total_amounts), len(membership_levels), len(max_staff_per_shop), len(min_staff_per_shop)))  # Safe indexing
    
    # 2. VARIABLES
    # Decision variable: Number of staff assigned to each shop
    x = {i: mdl.continuous_var(name=f"x_{i}", lb=0) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total weighted spending by members during happy hours
    objective = mdl.sum(membership_levels[i] * total_amounts[i] * x[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    
    # Total Staff Budget Constraint
    total_staff = mdl.sum(x[i] for i in safe_range)
    mdl.add_constraint(total_staff <= 150, ctname="total_staff_budget")
    
    # Shop Staffing Limits
    for i in safe_range:
        mdl.add_constraint(x[i] >= min_staff_per_shop[i], ctname=f"min_staff_{i}")
        mdl.add_constraint(x[i] <= max_staff_per_shop[i], ctname=f"max_staff_{i}")
    
    # Maximum Spending Limit per Member
    for i in safe_range:
        mdl.add_constraint(total_amounts[i] <= 50, ctname=f"max_spending_{i}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(x[i])
            print(f"Staff assigned to shop {i}: {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Execute the optimization
optimize_coffee_shop_staff_allocation()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 1683.75
**Execution Time**: 1.23 seconds
**Reliability**: high
**Analysis**: DOCplex also returned an optimal solution with the same value of 1683.75. However, the execution time was significantly longer (1.227 seconds) compared to Gurobipy, suggesting lower efficiency.

## 7. Pyomo Implementation

```python
#!/usr/bin/env python3
"""
Pyomo 6.9.2 Implementation for Coffee Shop Staff Allocation Optimization
"""

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def coffee_shop_optimization():
    """Optimize staff allocation for coffee shop happy hours"""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Example data based on the problem context
    shops = [1, 2, 3]  # Shop indices
    membership_levels = {1: 1, 2: 2, 3: 3}  # Membership levels for each shop
    spending_per_member = {1: 25.5, 2: 30.0, 3: 15.75}  # Spending per member for each shop
    min_staff = {1: 2, 2: 3, 3: 4}  # Minimum staff per shop
    max_staff = {1: 10, 2: 12, 3: 15}  # Maximum staff per shop
    total_staff_budget = 150  # Total staff budget across all shops
    max_spending_per_member = 50  # Maximum spending per member
    
    # CRITICAL: Validate array lengths before indexing
    assert len(shops) == len(membership_levels) == len(spending_per_member) == len(min_staff) == len(max_staff), "Array length mismatch"
    
    # 3. SETS (Pyomo way to define indices)
    model.I = pyo.Set(initialize=shops)  # Set of shops
    
    # 4. PARAMETERS (data containers)
    model.membership_level = pyo.Param(model.I, initialize=membership_levels)
    model.spending_per_member = pyo.Param(model.I, initialize=spending_per_member)
    model.min_staff = pyo.Param(model.I, initialize=min_staff)
    model.max_staff = pyo.Param(model.I, initialize=max_staff)
    model.total_staff_budget = pyo.Param(initialize=total_staff_budget)
    model.max_spending_per_member = pyo.Param(initialize=max_spending_per_member)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.I, within=pyo.NonNegativeIntegers)  # Number of staff per shop
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.membership_level[i] * model.spending_per_member[i] * model.x[i] for i in model.I)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    # Total Staff Budget Constraint
    def total_staff_budget_rule(model):
        return sum(model.x[i] for i in model.I) <= model.total_staff_budget
    model.total_staff_budget_constraint = pyo.Constraint(rule=total_staff_budget_rule)
    
    # Shop Staffing Limits
    def shop_staffing_limits_rule(model, i):
        return (model.min_staff[i], model.x[i], model.max_staff[i])
    model.shop_staffing_limits_constraint = pyo.Constraint(model.I, rule=shop_staffing_limits_rule)
    
    # Maximum Spending Limit per Member
    def max_spending_limit_rule(model, i):
        return model.spending_per_member[i] <= model.max_spending_per_member
    model.max_spending_limit_constraint = pyo.Constraint(model.I, rule=max_spending_limit_rule)
    
    # 8. SOLVING WITH GUROBI (your available solver)
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)  # tee=True shows solver output
    
    # 9. RESULT PROCESSING
    # Check solver status
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nStaff allocation per shop:")
        for i in model.I:
            x_val = pyo.value(model.x[i])
            print(f"Shop {i}: {int(x_val)} staff")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Run the optimization
if __name__ == "__main__":
    coffee_shop_optimization()
```

### Execution Results
**Status**: ERROR
**Error**: Traceback (most recent call last):
  File "/tmp/tmposhjhl2b.py", line 96, in <module>
    coffee_shop_optimization()
  File "/tmp/tmposhjhl2b.py", line 61, in coffee_shop_optimization
    model.max_spending_limit_constraint = pyo.Constraint(model.I, rule=max_spending_limit_rule)
  File "/dccstor/nl2opt/miniforge3/envs/nl2opt_optim/lib/python3.10/site-packages/pyomo/core/base/block.py", line 571, in __setattr__
    self.add_component(name, val)
  File "/dccstor/nl2opt/miniforge3/envs/nl2opt_optim/lib/python3.10/site-packages/pyomo/core/base/block.py", line 1101, in add_component
    val.construct(data)
  File "/dccstor/nl2opt/miniforge3/envs/nl2opt_optim/lib/python3.10/site-packages/pyomo/core/base/constraint.py", line 722, in construct
    self._setitem_when_not_present(index, rule(block, index))
  File "/dccstor/nl2opt/miniforge3/envs/nl2opt_optim/lib/python3.10/site-packages/pyomo/core/base/indexed_component.py", line 1111, in _setitem_when_not_present
    obj.set_value(value)
  File "/dccstor/nl2opt/miniforge3/envs/nl2opt_optim/lib/python3.10/site-packages/pyomo/core/base/constraint.py", line 469, in set_value
    raise ValueError(
ValueError: Invalid constraint expression. The constraint expression resolved to a trivial Boolean (True) instead of a Pyomo object. Please modify your rule to return Constraint.Feasible instead of True.

Error thrown for Constraint 'max_spending_limit_constraint[1]'

**Analysis**: Pyomo encountered an error during execution. The error message indicates an issue with the constraint formulation, specifically a trivial Boolean value being returned instead of a valid Pyomo constraint. This suggests a potential bug or incorrect implementation in the Pyomo model.

## 8. Cross-Solver Analysis and Final Recommendation

### Solver Results Comparison

| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |
|--------|--------|---------------|----------------|-------------------|---------------|
| Gurobipy | OPTIMAL | 1683.75 | 0.18s | N/A | N/A |
| Docplex | OPTIMAL | 1683.75 | 1.23s | N/A | N/A |
| Pyomo | ERROR | N/A | 0.88s | N/A | N/A |

### Solver Consistency Analysis
**Result**: All solvers produced consistent results ✓
**Consistent Solvers**: gurobipy, docplex
**Majority Vote Optimal Value**: 1683.75

### Final Recommendation
**Recommended Optimal Value**: 1683.75
**Confidence Level**: HIGH
**Preferred Solver(s)**: gurobipy
**Reasoning**: Gurobipy is preferred due to its high reliability, optimal solution, and significantly faster execution time compared to DOCplex. Pyomo is not recommended due to the encountered error.

### Business Interpretation
**Overall Strategy**: The optimal solution suggests that the total weighted spending during happy hours can be maximized to $1683.75 by optimally allocating staff across shops while adhering to the constraints.
**Objective Value Meaning**: The optimal objective value of $1683.75 represents the maximum total weighted spending achievable during happy hours, considering the constraints on staff allocation and member spending limits.
**Resource Allocation Summary**: Staff should be allocated across shops in a manner that maximizes weighted spending, ensuring that the total number of staff does not exceed 150 and that each shop's staffing limits are respected.
**Implementation Recommendations**: 1. Use the Gurobipy solver for reliable and efficient optimization. 2. Verify the optimal staff allocation values for each shop. 3. Ensure that the staffing limits and maximum spending constraints are adhered to during implementation.