# Complete GUROBIPY implementation - Retry Attempt 1

import gurobipy as gp
from gurobipy import GRB

def optimize_employee_assignment():
    # 1. MODEL & DATA SETUP
    model = gp.Model("employee_assignment_optimization")
    
    # Example data (replace with actual data from your database)
    employees = [1, 2, 3]
    companies = [1, 2, 3]
    
    # Years of experience for each employee
    experience = {1: 5, 2: 10, 3: 15}
    
    # Financial performance (profit) for each company
    profit = {1: 100000, 2: 200000, 3: 300000}
    
    # Maximum number of employees each company can hire
    max_employees = {1: 10, 2: 15, 3: 20}
    
    # Weights for the objective function
    weight_experience = 0.7
    weight_profit = 0.3
    
    # CRITICAL: Validate array lengths before loops
    assert len(employees) > 0, "No employees provided"
    assert len(companies) > 0, "No companies provided"
    assert len(experience) == len(employees), "Experience data mismatch"
    assert len(profit) == len(companies), "Profit data mismatch"
    assert len(max_employees) == len(companies), "Max employees data mismatch"
    
    # 2. VARIABLES
    # Binary decision variables: x[i,j] = 1 if employee i is assigned to company j
    x = model.addVars(employees, companies, vtype=GRB.BINARY, name="x")
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total employee experience score
    model.setObjective(
        gp.quicksum(
            (weight_experience * experience[i] + weight_profit * profit[j]) * x[i, j]
            for i in employees
            for j in companies
        ),
        GRB.MAXIMIZE
    )
    
    # 4. CONSTRAINTS
    
    # Employee Assignment Constraint: Each employee must be assigned to exactly one company
    for i in employees:
        model.addConstr(
            gp.quicksum(x[i, j] for j in companies) == 1,
            name=f"employee_assignment_{i}"
        )
    
    # Company Capacity Constraint: The total number of employees assigned to each company cannot exceed its maximum capacity
    for j in companies:
        model.addConstr(
            gp.quicksum(x[i, j] for i in employees) <= max_employees[j],
            name=f"company_capacity_{j}"
        )
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in employees:
            for j in companies:
                if x[i, j].x > 0.5:
                    print(f"Employee {i} assigned to Company {j}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Execute the optimization
optimize_employee_assignment()