# Complete PYOMO implementation - Retry Attempt 4

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def culture_company_optimization():
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    profit_book_club = 0.15
    profit_movie = 0.20
    budget = 1000000
    min_book_club = 100000
    min_movie = 150000
    max_book_club = 500000
    max_movie = 600000
    min_total_profit = 200000
    
    # 3. VARIABLES
    model.x_b = pyo.Var(within=pyo.NonNegativeReals, bounds=(min_book_club, max_book_club))
    model.x_m = pyo.Var(within=pyo.NonNegativeReals, bounds=(min_movie, max_movie))
    
    # 4. OBJECTIVE FUNCTION
    def obj_rule(model):
        return profit_book_club * model.x_b + profit_movie * model.x_m
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 5. CONSTRAINTS
    def budget_constraint_rule(model):
        return model.x_b + model.x_m <= budget
    model.budget_constraint = pyo.Constraint(rule=budget_constraint_rule)
    
    def min_profit_constraint_rule(model):
        return profit_book_club * model.x_b + profit_movie * model.x_m >= min_total_profit
    model.min_profit_constraint = pyo.Constraint(rule=min_profit_constraint_rule)
    
    # 6. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)  # tee=True shows solver output
    
    # 7. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nVariable values:")
        print(f"x_b (Book Clubs Investment): {pyo.value(model.x_b)}")
        print(f"x_m (Movies Investment): {pyo.value(model.x_m)}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Execute the optimization
culture_company_optimization()