# Complete DOCPLEX implementation - Retry Attempt 1

from docplex.mp.model import Model

def decoration_competition_optimization():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="decoration_competition")
    
    # Data from the database
    members = [1, 2, 3]
    rounds = [1, 2, 3]
    
    # Ranking points for each member in each round
    rank_in_round = {
        (1, 1): 15,
        (2, 2): 20,
        (3, 3): 25
    }
    
    # Member participation limit
    member_limit = {m: 2 for m in members}
    
    # Round participant requirement
    round_requirement = {r: 2 for r in rounds}
    
    # 2. VARIABLES
    # Binary decision variable: x[m, r] = 1 if member m is assigned to round r, 0 otherwise
    x = mdl.binary_var_dict(keys=[(m, r) for m in members for r in rounds], name="x")
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total ranking points
    objective = mdl.sum(rank_in_round[m, r] * x[m, r] for m in members for r in rounds)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    
    # Member Participation Limit: Each member can participate in at most 2 rounds
    for m in members:
        mdl.add_constraint(mdl.sum(x[m, r] for r in rounds) <= member_limit[m], ctname=f"member_limit_{m}")
    
    # Round Participant Requirement: Each round must have exactly 2 participants
    for r in rounds:
        mdl.add_constraint(mdl.sum(x[m, r] for m in members) == round_requirement[r], ctname=f"round_requirement_{r}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for m in members:
            for r in rounds:
                if solution.get_value(x[m, r]) > 0.5:
                    print(f"Member {m} assigned to Round {r}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Execute the optimization
decoration_competition_optimization()