#!/usr/bin/env python3
"""
DOCplex 2.29.245 Implementation for Entrepreneur Investment Optimization
"""

from docplex.mp.model import Model

def optimize_investment():
    """Optimize investment allocation across entrepreneurs to maximize return."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="entrepreneur_investment")
    
    # Data from the entrepreneurs table
    entrepreneur_ids = [1, 2, 3]
    return_rates = [0.12, 0.18, 0.1]
    initial_investments = [150000, 200000, 100000]
    
    # Business configuration parameters
    total_budget = 1000000
    individual_investment_limit = 200000
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(entrepreneur_ids) == len(return_rates) == len(initial_investments), "Array length mismatch"
    safe_range = range(min(len(entrepreneur_ids), len(return_rates), len(initial_investments)))  # Safe indexing
    
    # 2. VARIABLES
    # Decision variables: amount allocated to each entrepreneur
    x = {i: mdl.continuous_var(name=f"x_{entrepreneur_ids[i]}", lb=0) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total expected return on investment
    objective = mdl.sum(return_rates[i] * x[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    
    # Total Budget Constraint
    total_investment = mdl.sum(x[i] for i in safe_range)
    mdl.add_constraint(total_investment <= total_budget, ctname="total_budget")
    
    # Individual Investment Limit
    for i in safe_range:
        mdl.add_constraint(x[i] <= individual_investment_limit, ctname=f"individual_limit_{entrepreneur_ids[i]}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(x[i])
            print(f"Allocated to entrepreneur {entrepreneur_ids[i]}: ${value:.2f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Execute the optimization
if __name__ == "__main__":
    optimize_investment()