#!/usr/bin/env python3
"""
Gurobipy Implementation for Entrepreneur Investment Optimization
"""

import gurobipy as gp
from gurobipy import GRB

def optimize_investment():
    # 1. MODEL & DATA SETUP
    model = gp.Model("entrepreneur_investment")
    
    # Data from the entrepreneurs table
    entrepreneur_ids = [1, 2, 3]
    return_rates = [0.12, 0.18, 0.1]
    initial_investments = [150000, 200000, 100000]
    
    # Business configuration parameters
    total_budget = 1000000
    individual_investment_limit = 200000
    
    # CRITICAL: Validate array lengths before loops
    assert len(entrepreneur_ids) == len(return_rates) == len(initial_investments), "Array length mismatch"
    
    # 2. VARIABLES
    # Decision variables: amount allocated to each entrepreneur
    x = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"x_{i}", lb=0) 
         for i in entrepreneur_ids}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total expected return on investment
    model.setObjective(gp.quicksum(return_rates[i-1] * x[i] for i in entrepreneur_ids), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    
    # Total Budget Constraint
    model.addConstr(gp.quicksum(x[i] for i in entrepreneur_ids) <= total_budget, name="total_budget")
    
    # Individual Investment Limit
    for i in entrepreneur_ids:
        model.addConstr(x[i] <= individual_investment_limit, name=f"individual_limit_{i}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in entrepreneur_ids:
            if x[i].x > 1e-6:
                print(f"Allocation to entrepreneur {i}: {x[i].x:.2f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Execute the optimization
if __name__ == "__main__":
    optimize_investment()