# Complete DOCPLEX implementation - Retry Attempt 4

from docplex.mp.model import Model

def flight_assignment_optimization():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="flight_assignment_optimization")
    
    # Data from the problem
    airlines = [1, 2, 3]
    routes = [1, 2, 3]
    
    # Cost data
    cost_data = [
        (1, 1, 1200.0),
        (1, 2, 1800.0),
        (1, 3, 2500.0)
    ]
    
    # Capacity data
    capacity_data = [
        (1, 8),
        (2, 12),
        (3, 6)
    ]
    
    # Validate array lengths
    assert len(cost_data) == len(airlines) * len(routes), "Cost data length mismatch"
    assert len(capacity_data) == len(airlines), "Capacity data length mismatch"
    
    # Create dictionaries for cost and capacity
    cost = {(a, r): c for a, r, c in cost_data}
    capacity = {a: c for a, c in capacity_data}
    
    # 2. VARIABLES
    x = mdl.binary_var_dict(keys=[(a, r) for a in airlines for r in routes], name="x")
    
    # 3. OBJECTIVE FUNCTION
    total_cost = mdl.sum(cost[(a, r)] * x[(a, r)] for a in airlines for r in routes)
    mdl.minimize(total_cost)
    
    # 4. CONSTRAINTS
    
    # Route Coverage: Each route must be assigned to exactly one airline
    for r in routes:
        mdl.add_constraint(mdl.sum(x[(a, r)] for a in airlines) == 1, ctname=f"route_coverage_{r}")
    
    # Airline Capacity: No airline can be assigned more routes than its capacity
    for a in airlines:
        mdl.add_constraint(mdl.sum(x[(a, r)] for r in routes) <= capacity[a], ctname=f"capacity_{a}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for a in airlines:
            for r in routes:
                if solution.get_value(x[(a, r)]) > 0:
                    print(f"Airline {a} assigned to route {r}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Execute the optimization
flight_assignment_optimization()