# Complete DOCPLEX implementation - Retry Attempt 2

from docplex.mp.model import Model

def optimize_reservations():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="hotel_reservation_optimization")
    
    # Sample data based on the provided schema and realistic data
    rates = [120.0, 100.0, 150.0, 110.0, 130.0]
    guests = [2, 3, 1, 2, 4]  # Assuming number of guests per reservation
    overlapping_pairs = [(0, 1), (1, 2), (3, 4)]  # Assuming overlapping reservation pairs
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(rates) == len(guests), "Array length mismatch"
    safe_range = range(min(len(rates), len(guests)))  # Safe indexing
    
    # 2. VARIABLES
    x = {i: mdl.binary_var(name=f"x_{i}") for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(rates[i] * x[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    
    # Room Capacity Constraint
    total_guests = mdl.sum(guests[i] * x[i] for i in safe_range)
    mdl.add_constraint(total_guests <= 4, ctname="room_capacity")
    
    # Date Overlap Constraint
    for pair in overlapping_pairs:
        i, j = pair
        mdl.add_constraint(x[i] + x[j] <= 1, ctname=f"no_overlap_{i}_{j}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(x[i])
            if value > 1e-6:
                print(f"Reservation {i} accepted: {value:.0f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Execute the optimization
optimize_reservations()