#!/usr/bin/env python3
"""
Gurobipy implementation for medicine_enzyme_interaction optimization problem
"""

import gurobipy as gp
from gurobipy import GRB

def optimize_medicine_selection():
    """Optimize medicine selection for clinical trial"""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("medicine_selection")
    
    # Data from the problem
    medicine_ids = [1, 2, 3]
    effectiveness_scores = {1: 0.85, 2: 0.75, 3: 0.9}
    adverse_interaction_scores = {1: 0.15, 2: 0.25, 3: 0.1}
    fda_approved = {1: True, 2: False, 3: True}
    
    # Validate array lengths
    assert len(medicine_ids) == len(effectiveness_scores) == len(adverse_interaction_scores) == len(fda_approved), "Array length mismatch"
    
    # 2. VARIABLES
    x = {i: model.addVar(vtype=GRB.BINARY, name=f"x_{i}") for i in medicine_ids}
    
    # 3. OBJECTIVE FUNCTION
    model.setObjective(gp.quicksum(effectiveness_scores[i] * x[i] for i in medicine_ids), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    # Adverse Interaction Limit
    model.addConstr(gp.quicksum(adverse_interaction_scores[i] * x[i] for i in medicine_ids) <= 0.5, name="adverse_interaction_limit")
    
    # Medicine Count Limit
    model.addConstr(gp.quicksum(x[i] for i in medicine_ids) <= 3, name="medicine_count_limit")
    
    # FDA Approval Requirement
    for i in medicine_ids:
        if not fda_approved[i]:
            model.addConstr(x[i] == 0, name=f"fda_approval_{i}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in medicine_ids:
            if x[i].x > 1e-6:
                print(f"Medicine {i} is selected: {x[i].x:.0f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Execute the optimization
optimize_medicine_selection()