# Complete GUROBIPY implementation - Retry Attempt 1

import gurobipy as gp
from gurobipy import GRB

def optimize_movie_recommendations():
    # 1. MODEL & DATA SETUP
    model = gp.Model("movie_recommendation_optimization")
    
    # Data from the database
    movie_ids = [1, 2, 3]
    reviewer_ids = [1, 2, 3]
    
    # Star ratings
    star_ratings = {
        (1, 1): 5,
        (2, 2): 4,
        (3, 3): 3
    }
    
    # Movie recommendation limits
    movie_limits = {
        1: 10,
        2: 7,
        3: 5
    }
    
    # Reviewer recommendation limits
    reviewer_limits = {
        1: 5,
        2: 4,
        3: 3
    }
    
    # 2. VARIABLES
    x = model.addVars(reviewer_ids, movie_ids, vtype=GRB.INTEGER, name="x", lb=0)
    
    # 3. OBJECTIVE FUNCTION
    model.setObjective(
        gp.quicksum(star_ratings[i, j] * x[i, j] for i in reviewer_ids for j in movie_ids if (i, j) in star_ratings),
        GRB.MAXIMIZE
    )
    
    # 4. CONSTRAINTS
    
    # Movie Recommendation Limits
    for j in movie_ids:
        model.addConstr(
            gp.quicksum(x[i, j] for i in reviewer_ids) <= movie_limits[j],
            name=f"movie_limit_{j}"
        )
    
    # Reviewer Recommendation Limits
    for i in reviewer_ids:
        model.addConstr(
            gp.quicksum(x[i, j] for j in movie_ids) <= reviewer_limits[i],
            name=f"reviewer_limit_{i}"
        )
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in reviewer_ids:
            for j in movie_ids:
                if x[i, j].x > 1e-6:
                    print(f"x[{i},{j}] = {x[i, j].x}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Execute the optimization
optimize_movie_recommendations()