#!/usr/bin/env python3
"""
Gurobipy 12.0.2 Implementation for Pet Care Resource Allocation
"""

import gurobipy as gp
from gurobipy import GRB

def optimize_pet_care():
    """Optimize pet care resource allocation using Gurobipy"""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("pet_care_optimization")
    
    # Data from the database
    pet_type_costs = {'dog': 50.0, 'cat': 30.0, 'bird': 20.0}
    pet_care_requirements = {'dog': {'min_care': 10.0, 'max_care': 100.0},
                             'cat': {'min_care': 5.0, 'max_care': 50.0},
                             'bird': {'min_care': 3.0, 'max_care': 30.0}}
    resource_allocation = {'pet1': {'pet_type': 'dog', 'amount': 50.0},
                           'pet2': {'pet_type': 'cat', 'amount': 30.0},
                           'pet3': {'pet_type': 'bird', 'amount': 20.0}}
    
    # Business configuration parameter
    R = 200.0  # Maximum total resource allocation limit
    
    # Validate data consistency
    pets = list(resource_allocation.keys())
    assert len(pets) > 0, "No pets found in resource_allocation"
    
    # 2. VARIABLES
    x = {pet: model.addVar(vtype=GRB.CONTINUOUS, name=f"x_{pet}", lb=0) 
         for pet in pets}
    
    # 3. OBJECTIVE FUNCTION
    model.setObjective(
        gp.quicksum(pet_type_costs[resource_allocation[pet]['pet_type']] * x[pet] 
                    for pet in pets), 
        GRB.MINIMIZE
    )
    
    # 4. CONSTRAINTS
    
    # Minimum Care Requirement
    for pet in pets:
        pet_type = resource_allocation[pet]['pet_type']
        min_care = pet_care_requirements[pet_type]['min_care']
        model.addConstr(x[pet] >= min_care, name=f"min_care_{pet}")
    
    # Maximum Care Requirement
    for pet in pets:
        pet_type = resource_allocation[pet]['pet_type']
        max_care = pet_care_requirements[pet_type]['max_care']
        model.addConstr(x[pet] <= max_care, name=f"max_care_{pet}")
    
    # Total Resource Limit
    model.addConstr(gp.quicksum(x[pet] for pet in pets) <= R, name="total_resource_limit")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for pet in pets:
            if x[pet].x > 1e-6:
                print(f"Resources allocated to {pet}: {x[pet].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
if __name__ == "__main__":
    optimize_pet_care()