# Complete PYOMO implementation

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def roller_coaster_optimization():
    """Optimize roller coaster distribution across theme parks to maximize visitor satisfaction."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    parks = [1, 2, 3]
    coaster_types = ['Wooden', 'Steel', 'Inverted']
    
    # Visitor satisfaction scores
    satisfaction_scores = {
        (1, 'Wooden'): 8.5,
        (1, 'Steel'): 9.0,
        (1, 'Inverted'): 7.5,
        (2, 'Wooden'): 8.0,
        (2, 'Steel'): 9.5,
        (2, 'Inverted'): 8.0,
        (3, 'Wooden'): 7.0,
        (3, 'Steel'): 9.0,
        (3, 'Inverted'): 8.5
    }
    
    # Park budgets
    park_budgets = {1: 1000000, 2: 1500000, 3: 2000000}
    
    # Park available space
    park_space = {1: 10000, 2: 15000, 3: 20000}
    
    # Park maximum roller coasters
    park_max_coasters = {1: 5, 2: 7, 3: 10}
    
    # Fixed cost and space per roller coaster
    cost_per_coaster = 500000
    space_per_coaster = 2000
    
    # 3. SETS
    model.P = pyo.Set(initialize=parks)
    model.C = pyo.Set(initialize=coaster_types)
    
    # 4. PARAMETERS
    model.satisfaction = pyo.Param(model.P, model.C, initialize=satisfaction_scores)
    model.budget = pyo.Param(model.P, initialize=park_budgets)
    model.space = pyo.Param(model.P, initialize=park_space)
    model.max_coasters = pyo.Param(model.P, initialize=park_max_coasters)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.P, model.C, within=pyo.NonNegativeIntegers)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.satisfaction[p, c] * model.x[p, c] for p in model.P for c in model.C)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    
    # Budget Constraint
    def budget_rule(model, p):
        return sum(cost_per_coaster * model.x[p, c] for c in model.C) <= model.budget[p]
    model.budget_constraint = pyo.Constraint(model.P, rule=budget_rule)
    
    # Space Constraint
    def space_rule(model, p):
        return sum(space_per_coaster * model.x[p, c] for c in model.C) <= model.space[p]
    model.space_constraint = pyo.Constraint(model.P, rule=space_rule)
    
    # Maximum Roller Coasters Constraint
    def max_coasters_rule(model, p):
        return sum(model.x[p, c] for c in model.C) <= model.max_coasters[p]
    model.max_coasters_constraint = pyo.Constraint(model.P, rule=max_coasters_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)  # tee=True shows solver output
    
    # 9. RESULT PROCESSING
    # Check solver status
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nVariable values:")
        for p in model.P:
            for c in model.C:
                x_val = pyo.value(model.x[p, c])
                if x_val > 1e-6:  # Only print non-zero values
                    print(f"x[{p}, {c}] = {x_val:.0f}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Execute the optimization
if __name__ == "__main__":
    roller_coaster_optimization()