#!/usr/bin/env python3
"""
Gurobipy 12.0.2 Implementation for Shop Membership Optimization Problem
"""

import gurobipy as gp
from gurobipy import GRB

def optimize_shop_membership():
    """Optimize the allocation of membership benefits across branches to maximize customer satisfaction."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("shop_membership")
    
    # Data from the problem
    branches = [1, 2, 3]
    membership_amounts = {1: 5000.0, 2: 7500.0, 3: 10000.0}
    branch_capacities = {1: 100, 2: 150, 3: 200}
    total_budget = 500000
    
    # CRITICAL: Validate array lengths before loops
    assert len(branches) == len(membership_amounts) == len(branch_capacities), "Array length mismatch"
    
    # 2. VARIABLES
    # Decision variables: total pounds spent by members at each branch
    x = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"x_{i}", lb=0) for i in branches}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total pounds spent by members across all branches
    model.setObjective(gp.quicksum(x[i] for i in branches), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    
    # Constraint 1: Total Membership Benefits Budget
    # The sum of the membership amounts allocated to all branches must not exceed the total budget
    model.addConstr(gp.quicksum(membership_amounts[i] for i in branches) <= total_budget, name="total_budget")
    
    # Constraint 2: Branch Capacity Limits
    # The total pounds spent by members at each branch must not exceed the branch's capacity
    for i in branches:
        model.addConstr(x[i] <= branch_capacities[i], name=f"capacity_{i}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in branches:
            if x[i].x > 1e-6:
                print(f"x[{i}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Execute the optimization
if __name__ == "__main__":
    optimize_shop_membership()