#!/usr/bin/env python3
"""
Gurobipy 12.0.2 Implementation for Small Bank Optimization Problem
"""

import gurobipy as gp
from gurobipy import GRB

def small_bank_optimization():
    """Optimize fund allocation between savings and checking accounts to maximize interest income."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("small_bank_optimization")
    
    # Data from the problem
    customers = [1, 2, 3]
    total_funds = {1: 1000.0, 2: 1500.0, 3: 2000.0}
    savings_interest = 0.03
    checking_interest = 0.01
    min_savings = 100
    min_checking = 50
    max_savings = 50000
    max_checking = 30000
    total_bank_funds = 1000000
    
    # CRITICAL: Validate array lengths before loops
    assert len(customers) == len(total_funds), "Array length mismatch"
    
    # 2. VARIABLES
    s = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"s_{i}", lb=min_savings, ub=max_savings) 
         for i in customers}
    
    c = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"c_{i}", lb=min_checking, ub=max_checking) 
         for i in customers}
    
    # 3. OBJECTIVE FUNCTION
    model.setObjective(
        gp.quicksum(savings_interest * s[i] + checking_interest * c[i] for i in customers), 
        GRB.MAXIMIZE
    )
    
    # 4. CONSTRAINTS
    
    # Total Funds Constraint
    for i in customers:
        model.addConstr(s[i] + c[i] <= total_funds[i], name=f"total_funds_{i}")
    
    # Total Bank Funds Constraint
    model.addConstr(
        gp.quicksum(s[i] + c[i] for i in customers) <= total_bank_funds, 
        name="total_bank_funds"
    )
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in customers:
            print(f"Customer {i}: Savings = {s[i].x:.2f}, Checking = {c[i].x:.2f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Execute the optimization
if __name__ == "__main__":
    small_bank_optimization()